<?php
/**
 * Clase Solicitud - Manejo de solicitudes de certificados
 */

class Solicitud {
    private $db;
    private $auth;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }
    
    /**
     * Crear nueva solicitud
     */
    public function crear($datos, $archivos = []) {
        try {
            $this->db->beginTransaction();
            
            // Generar número de solicitud único
            $numeroSolicitud = $this->generarNumeroSolicitud();
            
            // Validar RUT
            if (!$this->validarRut($datos['rut'])) {
                throw new Exception('RUT inválido');
            }
            
            // Insertar solicitud
            $solicitudId = $this->db->insert('solicitudes', [
                'numero_solicitud' => $numeroSolicitud,
                'nombre_completo' => $datos['nombre_completo'],
                'rut' => $this->formatearRut($datos['rut']),
                'direccion' => $datos['direccion'],
                'telefono' => $datos['telefono'],
                'email' => $datos['email'],
                'motivo' => $datos['motivo'],
                'estado' => 'pendiente',
                'ip_solicitante' => $_SERVER['REMOTE_ADDR'] ?? null
            ]);
            
            // Procesar archivos adjuntos
            if (!empty($archivos)) {
                foreach ($archivos as $archivo) {
                    $this->guardarAdjunto($solicitudId, $archivo);
                }
            }
            
            $this->db->commit();
            
            // Registrar log
            $this->auth->logActivity(null, $solicitudId, 'solicitud_creada', 
                "Nueva solicitud creada: {$numeroSolicitud}");
            
            return [
                'success' => true,
                'solicitud_id' => $solicitudId,
                'numero_solicitud' => $numeroSolicitud,
                'message' => 'Solicitud creada exitosamente'
            ];
            
        } catch (Exception $e) {
            $this->db->rollback();
            error_log("Error al crear solicitud: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error al crear la solicitud: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Obtener todas las solicitudes con filtros
     */
    public function obtenerTodas($filtros = []) {
        $where = [];
        $params = [];
        
        if (!empty($filtros['estado'])) {
            $where[] = "s.estado = ?";
            $params[] = $filtros['estado'];
        }
        
        if (!empty($filtros['busqueda'])) {
            $where[] = "(s.numero_solicitud LIKE ? OR s.nombre_completo LIKE ? OR s.rut LIKE ?)";
            $busqueda = "%{$filtros['busqueda']}%";
            $params[] = $busqueda;
            $params[] = $busqueda;
            $params[] = $busqueda;
        }
        
        if (!empty($filtros['fecha_desde'])) {
            $where[] = "DATE(s.fecha_solicitud) >= ?";
            $params[] = $filtros['fecha_desde'];
        }
        
        if (!empty($filtros['fecha_hasta'])) {
            $where[] = "DATE(s.fecha_solicitud) <= ?";
            $params[] = $filtros['fecha_hasta'];
        }
        
        $whereClause = !empty($where) ? "WHERE " . implode(" AND ", $where) : "";
        
        $sql = "SELECT s.*, 
                       u.nombre_completo as procesado_por,
                       (SELECT COUNT(*) FROM adjuntos WHERE solicitud_id = s.id) as num_adjuntos,
                       (SELECT COUNT(*) FROM certificados WHERE solicitud_id = s.id) as tiene_certificado
                FROM solicitudes s
                LEFT JOIN usuarios u ON s.usuario_procesado = u.id
                {$whereClause}
                ORDER BY s.fecha_solicitud DESC";
        
        return $this->db->select($sql, $params);
    }
    
    /**
     * Obtener una solicitud por ID
     */
    public function obtenerPorId($id) {
        $sql = "SELECT s.*, 
                       u.nombre_completo as procesado_por,
                       u.cargo as cargo_procesador
                FROM solicitudes s
                LEFT JOIN usuarios u ON s.usuario_procesado = u.id
                WHERE s.id = ?";
        
        $solicitud = $this->db->selectOne($sql, [$id]);
        
        if ($solicitud) {
            // Obtener adjuntos
            $solicitud['adjuntos'] = $this->obtenerAdjuntos($id);
            
            // Obtener certificados
            $solicitud['certificados'] = $this->obtenerCertificados($id);
        }
        
        return $solicitud;
    }
    
    /**
     * Actualizar estado de solicitud
     */
    public function actualizarEstado($id, $estado, $observaciones = null) {
        $datos = [
            'estado' => $estado,
            'fecha_procesada' => date('Y-m-d H:i:s'),
            'usuario_procesado' => $this->auth->getUserId()
        ];
        
        if ($observaciones !== null) {
            $datos['observaciones'] = $observaciones;
        }
        
        $updated = $this->db->update('solicitudes', $datos, 'id = ?', [$id]);
        
        if ($updated) {
            $this->auth->logActivity(
                $this->auth->getUserId(), 
                $id, 
                'estado_actualizado', 
                "Estado cambiado a: {$estado}"
            );
        }
        
        return $updated > 0;
    }
    
    /**
     * Obtener adjuntos de una solicitud
     */
    public function obtenerAdjuntos($solicitudId) {
        $sql = "SELECT * FROM adjuntos WHERE solicitud_id = ? ORDER BY fecha_subida ASC";
        return $this->db->select($sql, [$solicitudId]);
    }
    
    /**
     * Obtener certificados de una solicitud
     */
    public function obtenerCertificados($solicitudId) {
        $sql = "SELECT c.*, u.nombre_completo as generado_por
                FROM certificados c
                LEFT JOIN usuarios u ON c.usuario_generador = u.id
                WHERE c.solicitud_id = ?
                ORDER BY c.fecha_generacion DESC";
        
        return $this->db->select($sql, [$solicitudId]);
    }
    
    /**
     * Guardar archivo adjunto
     */
    private function guardarAdjunto($solicitudId, $archivo) {
        // Crear directorio si no existe
        $targetDir = SOLICITUDES_PATH . '/' . $solicitudId;
        if (!is_dir($targetDir)) {
            mkdir($targetDir, 0755, true);
        }
        
        // Generar nombre único
        $extension = pathinfo($archivo['name'], PATHINFO_EXTENSION);
        $nombreArchivo = uniqid() . '_' . time() . '.' . $extension;
        $rutaCompleta = $targetDir . '/' . $nombreArchivo;
        
        // Mover archivo
        if (move_uploaded_file($archivo['tmp_name'], $rutaCompleta)) {
            // Guardar en base de datos
            $this->db->insert('adjuntos', [
                'solicitud_id' => $solicitudId,
                'nombre_archivo' => $archivo['name'],
                'ruta_archivo' => $rutaCompleta,
                'tipo_mime' => $archivo['type'],
                'tamano_bytes' => $archivo['size']
            ]);
            
            return true;
        }
        
        return false;
    }
    
    /**
     * Generar número de solicitud único
     */
    private function generarNumeroSolicitud() {
        $prefijo = 'SOL';
        $anio = date('Y');
        
        // Obtener el último número del año
        $sql = "SELECT numero_solicitud FROM solicitudes 
                WHERE numero_solicitud LIKE ? 
                ORDER BY id DESC LIMIT 1";
        
        $ultimo = $this->db->selectOne($sql, ["{$prefijo}-{$anio}-%"]);
        
        if ($ultimo) {
            $partes = explode('-', $ultimo['numero_solicitud']);
            $numero = intval($partes[2] ?? 0) + 1;
        } else {
            $numero = 1;
        }
        
        return sprintf("%s-%s-%05d", $prefijo, $anio, $numero);
    }
    
    /**
     * Validar RUT chileno
     */
    private function validarRut($rut) {
        $rut = preg_replace('/[^0-9kK]/', '', $rut);
        if (strlen($rut) < 2) return false;
        
        $dv = strtoupper(substr($rut, -1));
        $numero = substr($rut, 0, -1);
        
        $suma = 0;
        $multiplo = 2;
        
        for ($i = strlen($numero) - 1; $i >= 0; $i--) {
            $suma += $numero[$i] * $multiplo;
            $multiplo = $multiplo < 7 ? $multiplo + 1 : 2;
        }
        
        $dvEsperado = 11 - ($suma % 11);
        $dvEsperado = $dvEsperado == 11 ? '0' : ($dvEsperado == 10 ? 'K' : (string)$dvEsperado);
        
        return $dv === $dvEsperado;
    }
    
    /**
     * Formatear RUT
     */
    private function formatearRut($rut) {
        $rut = preg_replace('/[^0-9kK]/', '', $rut);
        $dv = substr($rut, -1);
        $numero = substr($rut, 0, -1);
        return number_format($numero, 0, '', '.') . '-' . $dv;
    }
    
    /**
     * Obtener estadísticas
     */
    public function obtenerEstadisticas() {
        $stats = [];
        
        // Total de solicitudes
        $sql = "SELECT COUNT(*) as total FROM solicitudes";
        $result = $this->db->selectOne($sql);
        $stats['total'] = $result['total'];
        
        // Por estado
        $sql = "SELECT estado, COUNT(*) as cantidad FROM solicitudes GROUP BY estado";
        $estados = $this->db->select($sql);
        foreach ($estados as $estado) {
            $stats['por_estado'][$estado['estado']] = $estado['cantidad'];
        }
        
        // Del mes actual
        $sql = "SELECT COUNT(*) as total FROM solicitudes 
                WHERE MONTH(fecha_solicitud) = MONTH(CURRENT_DATE()) 
                AND YEAR(fecha_solicitud) = YEAR(CURRENT_DATE())";
        $result = $this->db->selectOne($sql);
        $stats['mes_actual'] = $result['total'];
        
        return $stats;
    }
}
