<?php
use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;

require_once BASE_PATH . '/vendor/autoload.php';

/**
 * Clase Mailer - Envío de correos electrónicos
 */

class Mailer {
    private $db;
    private $auth;
    private $mailer;
    private $config;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
        $this->config = $this->obtenerConfiguracion();
        $this->inicializarMailer();
    }
    
    /**
     * Inicializar PHPMailer
     */
    private function inicializarMailer() {
        $this->mailer = new PHPMailer(true);
        
        try {
            // Configuración del servidor SMTP
            $this->mailer->isSMTP();
            $this->mailer->Host = $this->config['smtp_host'];
            $this->mailer->SMTPAuth = true;
            $this->mailer->Username = $this->config['smtp_user'];
            $this->mailer->Password = $this->config['smtp_password'];
            $this->mailer->SMTPSecure = $this->config['smtp_secure'];
            $this->mailer->Port = $this->config['smtp_port'];
            $this->mailer->CharSet = 'UTF-8';
            
            // Remitente
            $this->mailer->setFrom(
                $this->config['smtp_from_email'],
                $this->config['smtp_from_name']
            );
            
        } catch (Exception $e) {
            error_log("Error al inicializar PHPMailer: " . $e->getMessage());
        }
    }
    
    /**
     * Enviar confirmación de solicitud
     */
    public function enviarConfirmacionSolicitud($solicitudId) {
        try {
            // Obtener datos de la solicitud
            $solicitud = new Solicitud();
            $datos = $solicitud->obtenerPorId($solicitudId);
            
            if (!$datos) {
                throw new Exception('Solicitud no encontrada');
            }
            
            // Configurar destinatario
            $this->mailer->addAddress($datos['email'], $datos['nombre_completo']);
            
            // Asunto
            $this->mailer->Subject = 'Confirmación de Solicitud de Certificado de Residencia';
            
            // Cuerpo del mensaje
            $this->mailer->isHTML(true);
            $this->mailer->Body = $this->templateConfirmacionSolicitud($datos);
            $this->mailer->AltBody = strip_tags($this->mailer->Body);
            
            // Enviar
            $enviado = $this->mailer->send();
            
            if ($enviado) {
                $this->auth->logActivity(
                    null,
                    $solicitudId,
                    'email_confirmacion_enviado',
                    "Correo de confirmación enviado a: {$datos['email']}"
                );
            }
            
            return $enviado;
            
        } catch (Exception $e) {
            error_log("Error al enviar correo de confirmación: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Enviar certificado por correo
     */
    public function enviarCertificado($certificadoId) {
        try {
            // Obtener datos del certificado
            $sql = "SELECT c.*, s.nombre_completo, s.email, s.rut
                    FROM certificados c
                    INNER JOIN solicitudes s ON c.solicitud_id = s.id
                    WHERE c.id = ?";
            
            $certificado = $this->db->selectOne($sql, [$certificadoId]);
            
            if (!$certificado) {
                throw new Exception('Certificado no encontrado');
            }
            
            if (!file_exists($certificado['ruta_pdf'])) {
                throw new Exception('Archivo PDF no encontrado');
            }
            
            // Configurar destinatario
            $this->mailer->addAddress($certificado['email'], $certificado['nombre_completo']);
            
            // Adjuntar PDF
            $this->mailer->addAttachment(
                $certificado['ruta_pdf'],
                'Certificado_Residencia_' . $certificado['numero_certificado'] . '.pdf'
            );
            
            // Asunto
            $this->mailer->Subject = 'Certificado de Residencia - ' . $certificado['numero_certificado'];
            
            // Cuerpo del mensaje
            $this->mailer->isHTML(true);
            $this->mailer->Body = $this->templateEnvioCertificado($certificado);
            $this->mailer->AltBody = strip_tags($this->mailer->Body);
            
            // Enviar
            $enviado = $this->mailer->send();
            
            if ($enviado) {
                // Actualizar base de datos
                $this->db->update('certificados',
                    [
                        'enviado_correo' => 1,
                        'fecha_envio_correo' => date('Y-m-d H:i:s')
                    ],
                    'id = ?',
                    [$certificadoId]
                );
                
                // Actualizar estado de solicitud
                $solicitud = new Solicitud();
                $solicitud->actualizarEstado($certificado['solicitud_id'], 'enviada');
                
                // Registrar log
                $this->auth->logActivity(
                    $this->auth->getUserId(),
                    $certificado['solicitud_id'],
                    'certificado_enviado',
                    "Certificado enviado por correo a: {$certificado['email']}"
                );
            }
            
            return [
                'success' => $enviado,
                'message' => $enviado ? 'Certificado enviado exitosamente' : 'Error al enviar el certificado'
            ];
            
        } catch (Exception $e) {
            error_log("Error al enviar certificado: " . $e->getMessage());
            return [
                'success' => false,
                'message' => 'Error al enviar el certificado: ' . $e->getMessage()
            ];
        }
    }
    
    /**
     * Template de confirmación de solicitud
     */
    private function templateConfirmacionSolicitud($datos) {
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background-color: #0066cc; color: white; padding: 20px; text-align: center; }
                .content { background-color: #f9f9f9; padding: 20px; margin: 20px 0; }
                .footer { text-align: center; font-size: 12px; color: #666; padding: 20px; }
                .info-box { background-color: white; padding: 15px; margin: 10px 0; border-left: 4px solid #0066cc; }
                .btn { display: inline-block; padding: 10px 20px; background-color: #0066cc; color: white; text-decoration: none; border-radius: 5px; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>' . $this->config['nombre_municipio'] . '</h1>
                    <p>Sistema de Certificados de Residencia</p>
                </div>
                
                <div class="content">
                    <h2>Solicitud Recibida Exitosamente</h2>
                    <p>Estimado/a <strong>' . htmlspecialchars($datos['nombre_completo']) . '</strong>,</p>
                    <p>Hemos recibido su solicitud de certificado de residencia con los siguientes datos:</p>
                    
                    <div class="info-box">
                        <p><strong>Número de Solicitud:</strong> ' . $datos['numero_solicitud'] . '</p>
                        <p><strong>RUT:</strong> ' . $datos['rut'] . '</p>
                        <p><strong>Fecha de Solicitud:</strong> ' . date('d/m/Y H:i', strtotime($datos['fecha_solicitud'])) . '</p>
                        <p><strong>Estado:</strong> Pendiente de revisión</p>
                    </div>
                    
                    <p>Su solicitud será procesada en un plazo máximo de 48 horas hábiles. Recibirá un nuevo correo cuando su certificado esté listo.</p>
                    
                    <p><strong>Importante:</strong> Guarde el número de solicitud para futuras consultas.</p>
                </div>
                
                <div class="footer">
                    <p>' . $this->config['direccion_municipio'] . '</p>
                    <p>Teléfono: ' . $this->config['telefono_municipio'] . '</p>
                    <p>Email: ' . $this->config['email_municipio'] . '</p>
                    <p style="margin-top: 20px; font-size: 11px;">
                        Este es un correo automático, por favor no responder.
                    </p>
                </div>
            </div>
        </body>
        </html>';
        
        return $html;
    }
    
    /**
     * Template de envío de certificado
     */
    private function templateEnvioCertificado($certificado) {
        $html = '
        <!DOCTYPE html>
        <html>
        <head>
            <meta charset="UTF-8">
            <style>
                body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
                .container { max-width: 600px; margin: 0 auto; padding: 20px; }
                .header { background-color: #28a745; color: white; padding: 20px; text-align: center; }
                .content { background-color: #f9f9f9; padding: 20px; margin: 20px 0; }
                .footer { text-align: center; font-size: 12px; color: #666; padding: 20px; }
                .success-box { background-color: #d4edda; border: 1px solid #c3e6cb; padding: 15px; margin: 10px 0; border-radius: 5px; }
                .info-box { background-color: white; padding: 15px; margin: 10px 0; border-left: 4px solid #28a745; }
            </style>
        </head>
        <body>
            <div class="container">
                <div class="header">
                    <h1>' . $this->config['nombre_municipio'] . '</h1>
                    <p>Su Certificado de Residencia está Listo</p>
                </div>
                
                <div class="content">
                    <div class="success-box">
                        <h3 style="margin-top: 0; color: #155724;">✓ Certificado Generado Exitosamente</h3>
                    </div>
                    
                    <p>Estimado/a <strong>' . htmlspecialchars($certificado['nombre_completo']) . '</strong>,</p>
                    <p>Nos complace informarle que su certificado de residencia ha sido generado y está adjunto en este correo.</p>
                    
                    <div class="info-box">
                        <p><strong>Número de Certificado:</strong> ' . $certificado['numero_certificado'] . '</p>
                        <p><strong>RUT:</strong> ' . $certificado['rut'] . '</p>
                        <p><strong>Fecha de Emisión:</strong> ' . date('d/m/Y H:i', strtotime($certificado['fecha_generacion'])) . '</p>
                        <p><strong>Estado:</strong> ' . ($certificado['firmado'] ? 'Firmado Digitalmente' : 'Generado') . '</p>
                    </div>
                    
                    <p><strong>El certificado se encuentra adjunto en formato PDF.</strong></p>
                    
                    <p>Este documento tiene validez legal y puede ser utilizado para los trámites que requiera.</p>
                    
                    <p style="background-color: #fff3cd; border: 1px solid #ffc107; padding: 10px; border-radius: 5px;">
                        <strong>Nota:</strong> Por favor, descargue y guarde el certificado adjunto. Este correo es su comprobante de emisión.
                    </p>
                </div>
                
                <div class="footer">
                    <p>' . $this->config['direccion_municipio'] . '</p>
                    <p>Teléfono: ' . $this->config['telefono_municipio'] . '</p>
                    <p>Email: ' . $this->config['email_municipio'] . '</p>
                    <p style="margin-top: 20px; font-size: 11px;">
                        Este es un correo automático, por favor no responder.
                    </p>
                </div>
            </div>
        </body>
        </html>';
        
        return $html;
    }
    
    /**
     * Obtener configuración del sistema
     */
    private function obtenerConfiguracion() {
        $sql = "SELECT clave, valor FROM configuracion";
        $resultados = $this->db->select($sql);
        
        $config = [];
        foreach ($resultados as $row) {
            $config[$row['clave']] = $row['valor'];
        }
        
        return $config;
    }
}
