<?php
/**
 * Clase CertificadoPDF - Generación de certificados de residencia en PDF
 * Versión mejorada con mejor manejo de errores y múltiples métodos de generación
 */

class CertificadoPDF {
    private $db;
    private $auth;
    
    public function __construct() {
        $this->db = Database::getInstance();
        $this->auth = new Auth();
    }
    
    /**
     * Generar certificado de residencia
     * @param int $solicitudId ID de la solicitud
     * @param array $opcionesFirma [opcional] Array con datos de firma: ['certificado_pfx' => ruta, 'password' => contraseña]
     */
    public function generar($solicitudId, $opcionesFirma = null) {
        try {
            // Validar ID
            if (empty($solicitudId) || !is_numeric($solicitudId)) {
                throw new Exception('ID de solicitud inválido');
            }
            
            // Obtener datos de la solicitud
            $solicitud = new Solicitud();
            $datos = $solicitud->obtenerPorId($solicitudId);
            
            if (!$datos) {
                throw new Exception('Solicitud no encontrada con ID: ' . $solicitudId);
            }
            
            // Obtener configuración
            $config = $this->obtenerConfiguracion();
            
            if (empty($config)) {
                throw new Exception('No se pudo cargar la configuración del sistema');
            }
            
            // Generar número de certificado
            $numeroCertificado = $this->generarNumeroCertificado();
            
            // Crear directorio si no existe
            $rutaPdf = $this->getRutaPDF($solicitudId, $numeroCertificado);
            $dir = dirname($rutaPdf);
            
            if (!is_dir($dir)) {
                if (!mkdir($dir, 0755, true)) {
                    throw new Exception('No se pudo crear el directorio: ' . $dir);
                }
            }
            
            if (!is_writable($dir)) {
                throw new Exception('El directorio no tiene permisos de escritura: ' . $dir);
            }
            
            // Procesar firma si se proporcionó
            $datosFirma = null;
            if ($opcionesFirma && isset($opcionesFirma['certificado_pfx']) && isset($opcionesFirma['password'])) {
                $datosFirma = $this->validarCertificadoDigital(
                    $opcionesFirma['certificado_pfx'], 
                    $opcionesFirma['password']
                );
                
                // Agregar datos del certificado para el QR
                // Solo necesitamos el número, el hash NO va en el QR
                $datosFirma['certificado_data'] = [
                    'numero_certificado' => $numeroCertificado,
                    'solicitud_id' => $solicitudId
                ];
            }
            
            // Generar el PDF (una sola vez)
            $this->crearPDFConFirmaIntegrada($datos, $config, $numeroCertificado, $rutaPdf, $datosFirma);
            
            // Verificar que el archivo se creó
            if (!file_exists($rutaPdf)) {
                throw new Exception('El archivo PDF no se generó correctamente');
            }
            
            // Calcular hash del documento FINAL
            $hash = hash_file('sha256', $rutaPdf);
            
            if (!$hash) {
                $hash = hash('sha256', $numeroCertificado . time());
            }
            
            // Guardar en base de datos
            $certificadoId = $this->db->insert('certificados', [
                'solicitud_id' => $solicitudId,
                'numero_certificado' => $numeroCertificado,
                'ruta_pdf' => $rutaPdf,
                'hash_documento' => $hash,
                'firmado' => $datosFirma ? 1 : 0,
                'fecha_firma' => $datosFirma ? date('Y-m-d H:i:s') : null,
                'usuario_generador' => $this->auth->getUserId()
            ]);
            
            // Actualizar estado de solicitud
            $solicitud->actualizarEstado($solicitudId, 'aprobada');
            
            // Registrar log
            $logMsg = "Certificado generado: {$numeroCertificado}";
            if ($datosFirma) {
                $logMsg .= " y firmado por {$datosFirma['nombre']}";
            }
            
            $this->auth->logActivity(
                $this->auth->getUserId(),
                $solicitudId,
                $datosFirma ? 'certificado_generado_firmado' : 'certificado_generado',
                $logMsg
            );
            
            return [
                'success' => true,
                'certificado_id' => $certificadoId,
                'numero_certificado' => $numeroCertificado,
                'ruta_pdf' => $rutaPdf,
                'firmado' => $datosFirma ? true : false,
                'firmante' => $datosFirma ? $datosFirma['nombre'] : null,
                'message' => $datosFirma 
                    ? 'Certificado generado y firmado exitosamente' 
                    : 'Certificado generado exitosamente'
            ];
            
        } catch (Exception $e) {
            $errorMsg = $e->getMessage();
            $errorTrace = $e->getTraceAsString();
            
            error_log("=== ERROR AL GENERAR CERTIFICADO ===");
            error_log("Mensaje: " . $errorMsg);
            error_log("Archivo: " . $e->getFile() . " (Línea: " . $e->getLine() . ")");
            error_log("Stack Trace: " . $errorTrace);
            error_log("=====================================");
            
            return [
                'success' => false,
                'message' => $errorMsg,
                'error_file' => $e->getFile(),
                'error_line' => $e->getLine()
            ];
        }
    }
    
    /**
     * Validar certificado digital y extraer información
     */
    private function validarCertificadoDigital($rutaCertificado, $password) {
        // Verificar que el archivo existe
        if (!file_exists($rutaCertificado)) {
            throw new Exception('El archivo del certificado digital no existe');
        }
        
        // Leer contenido
        $certData = file_get_contents($rutaCertificado);
        if (!$certData) {
            throw new Exception('No se pudo leer el certificado digital');
        }
        
        // Validar contraseña y leer certificado
        $certs = [];
        if (!openssl_pkcs12_read($certData, $certs, $password)) {
            throw new Exception('❌ Contraseña incorrecta del certificado digital');
        }
        
        // Verificar datos necesarios
        if (!isset($certs['cert']) || !isset($certs['pkey'])) {
            throw new Exception('El certificado digital no contiene los datos necesarios');
        }
        
        // Extraer información
        $certInfo = openssl_x509_parse($certs['cert']);
        $nombreFirmante = $certInfo['subject']['CN'] ?? 'Firmante Desconocido';
        $emailFirmante = $certInfo['subject']['emailAddress'] ?? '';
        
        // Verificar vigencia
        if (time() > $certInfo['validTo_time_t']) {
            $validHasta = date('d/m/Y', $certInfo['validTo_time_t']);
            throw new Exception('El certificado digital ha expirado el ' . $validHasta);
        }
        
        return [
            'cert' => $certs['cert'],
            'pkey' => $certs['pkey'],
            'nombre' => $nombreFirmante,
            'email' => $emailFirmante,
            'valido_desde' => date('d/m/Y', $certInfo['validFrom_time_t']),
            'valido_hasta' => date('d/m/Y', $certInfo['validTo_time_t'])
        ];
    }
    
    /**
     * Crear PDF con firma integrada desde el inicio
     */
    private function crearPDFConFirmaIntegrada($datos, $config, $numeroCertificado, $rutaPdf, $datosFirma = null) {
        // Método 1: Usar TCPDF (preferido para firma integrada)
        if ($this->crearPDFConTCPDFIntegrado($datos, $config, $numeroCertificado, $rutaPdf, $datosFirma)) {
            return true;
        }
        
        // Método 2: Generar HTML y convertir con firma
        if ($this->crearPDFConDompdf($datos, $config, $numeroCertificado, $rutaPdf, $datosFirma)) {
            return true;
        }
        
        // Método 3: PDF básico sin firma
        return $this->crearPDFSimplificado($datos, $config, $numeroCertificado, $rutaPdf);
    }
    
    /**
     * Crear PDF con TCPDF con firma integrada
     */
    private function crearPDFConTCPDFIntegrado($datos, $config, $numeroCertificado, $rutaPdf, $datosFirma) {
        try {
            if (!file_exists(BASE_PATH . '/vendor/autoload.php')) {
                return false;
            }
            
            require_once BASE_PATH . '/vendor/autoload.php';
            
            if (!class_exists('TCPDF')) {
                return false;
            }
            
            $pdf = new TCPDF('P', 'mm', 'LETTER', true, 'UTF-8', false);
            
            // Configuración del documento
            $pdf->SetCreator('Sistema de Certificados de Residencia');
            $pdf->SetAuthor($config['nombre_municipio']);
            $pdf->SetTitle('Certificado de Residencia - ' . $numeroCertificado);
            
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->SetMargins(20, 20, 20);
            $pdf->SetAutoPageBreak(false); // IMPORTANTE: Desactivar auto page break para controlar
            $pdf->AddPage();
            
            // Generar contenido del certificado (optimizado para dejar espacio)
            $this->generarContenidoTCPDF($pdf, $datos, $config, $numeroCertificado);
            
            // Agregar firma visual si hay datos de firma
            // Los datos del certificado ya vienen en $datosFirma['certificado_data']
            if ($datosFirma) {
                $this->agregarFirmaVisualTCPDF($pdf, $datosFirma);
            }
            
            // Guardar
            $pdf->Output($rutaPdf, 'F');
            
            return file_exists($rutaPdf);
            
        } catch (Exception $e) {
            error_log("Error en crearPDFConTCPDFIntegrado: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Agregar firma visual profesional en la parte inferior izquierda del PDF
     */
    private function agregarFirmaVisualTCPDF($pdf, $datosFirma) {
        // NO cambiar la página actual, trabajar en la misma
        // Posicionar en la parte inferior de la página ACTUAL
        // Usar posición absoluta calculada desde el inicio de la página
        
        $pageHeight = $pdf->getPageHeight(); // Altura total de la página
        
        // Calcular posición para que quede 15mm del fondo
        $firmaAltura = 55; // Altura del cuadro de firma
        $margenInferior = 15; // Margen desde el fondo
        $startY = $pageHeight - $firmaAltura - $margenInferior;
        
        $startX = 20;
        $anchoTotal = 170;
        
        // CUADRO PRINCIPAL DE FIRMA (sin líneas que tapen)
        $pdf->SetFillColor(245, 248, 250); // Gris muy claro
        $pdf->SetDrawColor(200, 200, 200); // Gris para borde
        $pdf->SetLineWidth(0.3);
        $pdf->Rect($startX, $startY, $anchoTotal, $firmaAltura, 'DF');
        
        // ===== SECCIÓN SUPERIOR: LOGO Y TÍTULO =====
        
        // Logo de la firma (círculo con iniciales)
        $logoX = $startX + 8;
        $logoY = $startY + 8;
        $logoSize = 18; // Reducido de 20 a 18
        
        // Círculo con borde rojo/azul
        $pdf->SetFillColor(41, 128, 185); // Azul
        $pdf->SetDrawColor(231, 76, 60); // Rojo
        $pdf->SetLineWidth(1.5);
        $pdf->Circle($logoX + $logoSize/2, $logoY + $logoSize/2, $logoSize/2, 0, 360, 'FD');
        
        // Iniciales en el círculo
        $pdf->SetTextColor(255, 255, 255); // Blanco
        $pdf->SetFont('helvetica', 'B', 11);
        
        // Obtener iniciales del nombre
        $nombrePartes = explode(' ', $datosFirma['nombre']);
        $iniciales = '';
        foreach (array_slice($nombrePartes, 0, 2) as $parte) {
            $iniciales .= strtoupper(substr($parte, 0, 1));
        }
        if (strlen($iniciales) < 2) $iniciales = 'FD';
        
        $pdf->SetXY($logoX + 2, $logoY + 5);
        $pdf->Cell($logoSize - 4, 8, $iniciales, 0, 0, 'C');
        
        // Título "FIRMADO DIGITALMENTE"
        $pdf->SetTextColor(52, 73, 94);
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->SetXY($logoX + $logoSize + 4, $logoY + 2);
        $pdf->Cell(80, 5, 'FIRMADO DIGITALMENTE', 0, 1, 'L');
        
        // Subtítulo con fecha
        $pdf->SetFont('helvetica', '', 7);
        $pdf->SetTextColor(127, 140, 141);
        $pdf->SetX($logoX + $logoSize + 4);
        $pdf->Cell(80, 4, date('d/m/Y H:i:s'), 0, 1, 'L');
        
        // ===== SECCIÓN MEDIA: DATOS DEL FIRMANTE =====
        
        $infoY = $startY + 28;
        $pdf->SetTextColor(52, 73, 94);
        
        // Nombre
        $pdf->SetFont('helvetica', 'B', 7);
        $pdf->SetXY($startX + 5, $infoY);
        $pdf->Cell(25, 3.5, 'Nombre:', 0, 0, 'L');
        
        $pdf->SetFont('helvetica', '', 7);
        $nombreFirmante = $datosFirma['nombre'];
        if (strlen($nombreFirmante) > 35) {
            $nombreFirmante = substr($nombreFirmante, 0, 32) . '...';
        }
        $pdf->Cell(0, 3.5, $nombreFirmante, 0, 1, 'L');
        
        // Email
        if (!empty($datosFirma['email'])) {
            $pdf->SetFont('helvetica', 'B', 7);
            $pdf->SetX($startX + 5);
            $pdf->Cell(25, 3.5, 'Email:', 0, 0, 'L');
            
            $pdf->SetFont('helvetica', '', 7);
            $emailFirmante = $datosFirma['email'];
            if (strlen($emailFirmante) > 35) {
                $emailFirmante = substr($emailFirmante, 0, 32) . '...';
            }
            $pdf->Cell(0, 3.5, $emailFirmante, 0, 1, 'L');
        }
        
        // Fecha de firma
        $pdf->SetFont('helvetica', 'B', 7);
        $pdf->SetX($startX + 5);
        $pdf->Cell(25, 3.5, 'Fecha Firma:', 0, 0, 'L');
        
        $pdf->SetFont('helvetica', '', 7);
        $pdf->Cell(0, 3.5, date('d/m/Y H:i:s'), 0, 1, 'L');
        
        // Validez del certificado
        $pdf->SetFont('helvetica', 'B', 7);
        $pdf->SetX($startX + 5);
        $pdf->Cell(25, 3.5, 'Válido hasta:', 0, 0, 'L');
        
        $pdf->SetFont('helvetica', '', 7);
        $pdf->Cell(0, 3.5, $datosFirma['valido_hasta'], 0, 1, 'L');
        
        // ===== SECCIÓN DERECHA: QR Y VERIFICACIÓN =====
        
        // Posición del QR (derecha, sin que lo tape nada)
        $qrX = $startX + $anchoTotal - 32;
        $qrY = $startY + 8; // Más arriba para que no lo tape la línea
        $qrSize = 28; // Reducido de 30 a 28
        
        // Generar URL de verificación
        $certificadoActual = $datosFirma['certificado_data'] ?? [
            'numero_certificado' => 'TEMP',
            'hash_documento' => hash('sha256', time()),
            'id' => 0
        ];
        
        $urlVerificacion = $this->generarURLVerificacion($certificadoActual);
        
        // Generar código QR con TCPDF
        $pdf->write2DBarcode(
            $urlVerificacion,
            'QRCODE,L',
            $qrX,
            $qrY,
            $qrSize,
            $qrSize,
            array(
                'border' => false,
                'padding' => 0,
                'fgcolor' => array(0, 0, 0),
                'bgcolor' => array(255, 255, 255)
            ),
            'N'
        );
        
        // Texto de verificación junto al QR
        $pdf->SetFont('helvetica', 'B', 6);
        $pdf->SetTextColor(52, 73, 94);
        $pdf->SetXY($qrX - 42, $qrY + 2);
        $pdf->MultiCell(38, 3, 'Verifique este documento escaneando el código', 0, 'R');
        
        // ===== PIE DEL CUADRO: TOKEN Y HASH =====
        // NOTA: Línea separadora ELIMINADA para no tapar el QR
        
        $pieY = $startY + $firmaAltura - 8;
        
        // Token de verificación
        $pdf->SetFont('helvetica', '', 5.5);
        $pdf->SetTextColor(127, 140, 141);
        $pdf->SetXY($startX + 5, $pieY);
        
        $token = $this->generarTokenVerificacion($certificadoActual);
        $pdf->Cell(80, 2.5, 'Token: ' . $token, 0, 0, 'L');
        
        // Hash del documento (abreviado)
        $pdf->SetX($startX + 90);
        $hash = $certificadoActual['hash_documento'] ?? hash('sha256', time());
        $hashCorto = substr($hash, 0, 12) . '...' . substr($hash, -12);
        $pdf->Cell(0, 2.5, 'Hash: ' . $hashCorto, 0, 1, 'L');
        
        // Restaurar color de texto
        $pdf->SetTextColor(0, 0, 0);
    }
    
    /**
     * Obtener datos del certificado actual
     */
    private function obtenerCertificadoActual() {
        // Esta función será llamada durante la generación
        // Por ahora retornamos datos de ejemplo que se sobrescribirán
        return [
            'numero_certificado' => 'CERT-' . date('Y') . '-00000',
            'hash_documento' => hash('sha256', time()),
            'id' => 0
        ];
    }
    
    /**
     * Generar URL de verificación del certificado
     */
    private function generarURLVerificacion($certificado) {
        // Obtener configuración para el dominio
        $config = $this->obtenerConfiguracion();
        
        // URL base (ajustar según tu dominio)
        $baseUrl = $_SERVER['HTTP_HOST'] ?? 'localhost';
        $protocolo = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') ? 'https' : 'http';
        
        // Construir URL de verificación SOLO con número de certificado
        // El hash NO se incluye porque causaría problemas circulares
        // (el hash cambia al regenerar el PDF con el QR)
        $urlVerificacion = $protocolo . '://' . $baseUrl . '/public/verificar.php?cert=' . 
                          urlencode($certificado['numero_certificado']);
        
        return $urlVerificacion;
    }
    
    /**
     * Generar token único de verificación
     */
    private function generarTokenVerificacion($certificado) {
        // Generar token alfanumérico de 16 caracteres
        $data = $certificado['numero_certificado'] . $certificado['hash_documento'] . time();
        $hash = hash('sha256', $data);
        return strtoupper(substr($hash, 0, 16));
    }
    
    /**
     * Crear PDF con Dompdf
     */
    private function crearPDFConDompdf($datos, $config, $numeroCertificado, $rutaPdf, $datosFirma) {
        try {
            if (!file_exists(BASE_PATH . '/vendor/autoload.php')) {
                return false;
            }
            
            require_once BASE_PATH . '/vendor/autoload.php';
            
            if (!class_exists('\Dompdf\Dompdf')) {
                return false;
            }
            
            // Generar HTML con firma integrada
            $html = $this->generarHTMLConFirma($datos, $config, $numeroCertificado, $datosFirma);
            
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('letter', 'portrait');
            $dompdf->render();
            
            file_put_contents($rutaPdf, $dompdf->output());
            
            return file_exists($rutaPdf);
            
        } catch (Exception $e) {
            error_log("Error en crearPDFConDompdf: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Generar HTML del certificado con firma integrada
     */
    private function generarHTMLConFirma($datos, $config, $numeroCertificado, $datosFirma) {
        $usuario = $this->auth->getUser();
        $cargo = htmlspecialchars($usuario['cargo'] ?? 'Funcionario Municipal');
        $fecha = $this->fechaTexto(date('Y-m-d'));
        $nombreCompleto = htmlspecialchars($datos['nombre_completo']);
        $rut = htmlspecialchars($datos['rut']);
        $direccion = htmlspecialchars($datos['direccion']);
        $motivo = htmlspecialchars($datos['motivo']);
        $nombreMunicipio = htmlspecialchars($config['nombre_municipio']);
        $direccionMunicipio = htmlspecialchars($config['direccion_municipio']);
        $comuna = htmlspecialchars($config['comuna']);
        $region = htmlspecialchars($config['region']);
        $nombreUsuario = htmlspecialchars($usuario['nombre_completo']);
        
        // Sección de firma si está presente
        $seccionFirma = '';
        if ($datosFirma) {
            $nombreFirmante = htmlspecialchars($datosFirma['nombre']);
            $emailFirmante = htmlspecialchars($datosFirma['email']);
            $validoHasta = htmlspecialchars($datosFirma['valido_hasta']);
            $fechaHoraFirma = date('d/m/Y H:i:s');
            
            $seccionFirma = <<<FIRMA
    <div class="firma-digital">
        <div class="firma-header">🔒 FIRMA DIGITAL</div>
        <div class="firma-content">
            <p><strong>Firmado por:</strong> {$nombreFirmante}</p>
            <p><strong>Email:</strong> {$emailFirmante}</p>
            <p><strong>Fecha:</strong> {$fechaHoraFirma}</p>
            <p><strong>Válido hasta:</strong> {$validoHasta}</p>
            <p class="firma-verificado">✓ Firmado digitalmente</p>
        </div>
    </div>
FIRMA;
        }
        
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Certificado de Residencia - {$numeroCertificado}</title>
    <style>
        @page { margin: 2cm; }
        body { font-family: Arial, Helvetica, sans-serif; line-height: 1.6; color: #000; }
        .header { text-align: center; margin-bottom: 30px; padding-bottom: 20px; border-bottom: 2px solid #000; }
        .header h1 { margin: 10px 0; font-size: 18px; }
        .header p { margin: 5px 0; font-size: 11px; }
        .title { font-size: 16px; font-weight: bold; text-align: center; margin: 30px 0; text-decoration: underline; }
        .info { text-align: right; margin-bottom: 20px; font-size: 11px; }
        .content { text-align: justify; margin: 20px 0; }
        .data { margin: 20px 0; padding: 15px; background-color: #f5f5f5; }
        .data p { margin: 8px 0; }
        .data strong { display: inline-block; width: 120px; }
        .footer { margin-top: 60px; text-align: center; }
        .signature { margin-top: 40px; }
        .signature-line { border-top: 1px solid #000; width: 250px; margin: 0 auto; padding-top: 5px; }
        
        /* Estilos para firma digital */
        .firma-digital {
            position: absolute;
            bottom: 60px;
            left: 20px;
            width: 280px;
            border: 2px solid #0066cc;
            background-color: #f0f8ff;
            padding: 10px;
            font-size: 9px;
        }
        .firma-header {
            background-color: #0066cc;
            color: white;
            padding: 5px;
            text-align: center;
            font-weight: bold;
            margin: -10px -10px 10px -10px;
        }
        .firma-content p {
            margin: 4px 0;
            line-height: 1.4;
        }
        .firma-content strong {
            display: inline-block;
            width: 80px;
            font-size: 8px;
        }
        .firma-verificado {
            color: #009900;
            font-weight: bold;
            text-align: center;
            margin-top: 8px;
            font-size: 10px;
        }
    </style>
</head>
<body>
    <div class="header">
        <h1>{$nombreMunicipio}</h1>
        <p>{$direccionMunicipio}</p>
        <p>{$comuna} - {$region}</p>
    </div>
    
    <div class="title">CERTIFICADO DE RESIDENCIA</div>
    
    <div class="info">
        <p><strong>N°:</strong> {$numeroCertificado}</p>
        <p><strong>Fecha:</strong> {$comuna}, {$fecha}</p>
    </div>
    
    <div class="content">
        <p>El que suscribe, <strong>{$cargo}</strong> de la <strong>{$nombreMunicipio}</strong>, certifica que según los antecedentes disponibles en esta oficina:</p>
    </div>
    
    <div class="data">
        <p><strong>Nombre:</strong> {$nombreCompleto}</p>
        <p><strong>RUT:</strong> {$rut}</p>
        <p><strong>Domicilio:</strong> {$direccion}, {$comuna}</p>
    </div>
    
    <div class="content">
        <p><strong>TIENE SU RESIDENCIA</strong> en el domicilio antes indicado, perteneciente a la comuna de {$comuna}.</p>
        
        <p><strong>MOTIVO DE LA SOLICITUD:</strong><br>{$motivo}</p>
        
        <p>Se extiende el presente certificado a petición del interesado para los fines que estime conveniente.</p>
    </div>
    
    <div class="footer">
        <div class="signature">
            <div class="signature-line">
                <p><strong>{$nombreUsuario}</strong></p>
                <p>{$cargo}</p>
                <p>{$nombreMunicipio}</p>
            </div>
        </div>
        
        <p style="margin-top: 40px; font-size: 9px; color: #666;">
            Certificado N° {$numeroCertificado} - Fecha de emisión: {$fecha}
        </p>
    </div>
    
    {$seccionFirma}
</body>
</html>
HTML;
    }
    
    /**
     * Crear PDF simplificado usando HTML
     */
    private function crearPDFSimplificado($datos, $config, $numeroCertificado, $rutaPdf) {
        // Generar contenido HTML
        $html = $this->generarHTML($datos, $config, $numeroCertificado);
        
        // Método 1: Usar Dompdf si está disponible
        if ($this->intentarDompdf($html, $rutaPdf)) {
            return true;
        }
        
        // Método 2: Usar mPDF si está disponible
        if ($this->intentarMpdf($html, $rutaPdf)) {
            return true;
        }
        
        // Método 3: Usar TCPDF si está disponible
        if ($this->intentarTCPDF($datos, $config, $numeroCertificado, $rutaPdf)) {
            return true;
        }
        
        // Método 4: Guardar como HTML y crear PDF básico
        return $this->crearPDFBasico($datos, $config, $numeroCertificado, $rutaPdf, $html);
    }
    
    /**
     * Intentar generar con Dompdf
     */
    private function intentarDompdf($html, $rutaPdf) {
        try {
            if (!file_exists(BASE_PATH . '/vendor/autoload.php')) {
                return false;
            }
            
            require_once BASE_PATH . '/vendor/autoload.php';
            
            if (!class_exists('\Dompdf\Dompdf')) {
                return false;
            }
            
            $dompdf = new \Dompdf\Dompdf();
            $dompdf->loadHtml($html);
            $dompdf->setPaper('letter', 'portrait');
            $dompdf->render();
            
            file_put_contents($rutaPdf, $dompdf->output());
            
            return file_exists($rutaPdf);
            
        } catch (Exception $e) {
            error_log("Dompdf falló: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Intentar generar con mPDF
     */
    private function intentarMpdf($html, $rutaPdf) {
        try {
            if (!file_exists(BASE_PATH . '/vendor/autoload.php')) {
                return false;
            }
            
            require_once BASE_PATH . '/vendor/autoload.php';
            
            if (!class_exists('\Mpdf\Mpdf')) {
                return false;
            }
            
            $mpdf = new \Mpdf\Mpdf();
            $mpdf->WriteHTML($html);
            $mpdf->Output($rutaPdf, 'F');
            
            return file_exists($rutaPdf);
            
        } catch (Exception $e) {
            error_log("mPDF falló: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Intentar generar con TCPDF
     */
    private function intentarTCPDF($datos, $config, $numeroCertificado, $rutaPdf) {
        try {
            if (!file_exists(BASE_PATH . '/vendor/autoload.php')) {
                return false;
            }
            
            require_once BASE_PATH . '/vendor/autoload.php';
            
            if (!class_exists('TCPDF')) {
                return false;
            }
            
            $pdf = new TCPDF('P', 'mm', 'LETTER', true, 'UTF-8', false);
            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);
            $pdf->SetMargins(20, 20, 20);
            $pdf->AddPage();
            
            $this->generarContenidoTCPDF($pdf, $datos, $config, $numeroCertificado);
            
            $pdf->Output($rutaPdf, 'F');
            
            return file_exists($rutaPdf);
            
        } catch (Exception $e) {
            error_log("TCPDF falló: " . $e->getMessage());
            return false;
        }
    }
    
    /**
     * Crear PDF básico (fallback)
     */
    private function crearPDFBasico($datos, $config, $numeroCertificado, $rutaPdf, $html) {
        // Guardar HTML como referencia
        file_put_contents($rutaPdf . '.html', $html);
        
        // Crear un PDF básico manualmente
        $contenido = $this->generarContenidoTexto($datos, $config, $numeroCertificado);
        
        // Estructura PDF básica
        $pdf = "%PDF-1.4\n";
        $pdf .= "1 0 obj<</Type/Catalog/Pages 2 0 R>>endobj\n";
        $pdf .= "2 0 obj<</Type/Pages/Kids[3 0 R]/Count 1>>endobj\n";
        $pdf .= "3 0 obj<</Type/Page/Parent 2 0 R/MediaBox[0 0 612 792]/Contents 4 0 R/Resources<</Font<</F1<</Type/Font/Subtype/Type1/BaseFont/Helvetica>>>>>>endobj\n";
        
        $streamData = "BT /F1 12 Tf 50 750 Td (" . addslashes($contenido) . ") Tj ET";
        $streamLength = strlen($streamData);
        
        $pdf .= "4 0 obj<</Length $streamLength>>stream\n$streamData\nendstream endobj\n";
        $pdf .= "xref\n0 5\n0000000000 65535 f\n";
        $pdf .= str_pad(strlen("%PDF-1.4\n"), 10, "0", STR_PAD_LEFT) . " 00000 n\n";
        
        $xrefPos = strlen($pdf);
        $pdf .= "trailer<</Size 5/Root 1 0 R>>\nstartxref\n$xrefPos\n%%EOF";
        
        file_put_contents($rutaPdf, $pdf);
        
        return file_exists($rutaPdf);
    }
    
    /**
     * Generar HTML del certificado
     */
    private function generarHTML($datos, $config, $numeroCertificado) {
        $usuario = $this->auth->getUser();
        $cargo = htmlspecialchars($usuario['cargo'] ?? 'Funcionario Municipal');
        $fecha = $this->fechaTexto(date('Y-m-d'));
        $nombreCompleto = htmlspecialchars($datos['nombre_completo']);
        $rut = htmlspecialchars($datos['rut']);
        $direccion = htmlspecialchars($datos['direccion']);
        $motivo = htmlspecialchars($datos['motivo']);
        $nombreMunicipio = htmlspecialchars($config['nombre_municipio']);
        $direccionMunicipio = htmlspecialchars($config['direccion_municipio']);
        $comuna = htmlspecialchars($config['comuna']);
        $region = htmlspecialchars($config['region']);
        $nombreUsuario = htmlspecialchars($usuario['nombre_completo']);
        
        return <<<HTML
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Certificado de Residencia - {$numeroCertificado}</title>
    <style>
        @page { margin: 2cm; }
        body { font-family: Arial, Helvetica, sans-serif; line-height: 1.6; color: #000; }
        .header { text-align: center; margin-bottom: 30px; padding-bottom: 20px; border-bottom: 2px solid #000; }
        .header h1 { margin: 10px 0; font-size: 18px; }
        .header p { margin: 5px 0; font-size: 11px; }
        .title { font-size: 16px; font-weight: bold; text-align: center; margin: 30px 0; text-decoration: underline; }
        .info { text-align: right; margin-bottom: 20px; font-size: 11px; }
        .content { text-align: justify; margin: 20px 0; }
        .data { margin: 20px 0; padding: 15px; background-color: #f5f5f5; }
        .data p { margin: 8px 0; }
        .data strong { display: inline-block; width: 120px; }
        .footer { margin-top: 60px; text-align: center; }
        .signature { margin-top: 40px; }
        .signature-line { border-top: 1px solid #000; width: 250px; margin: 0 auto; padding-top: 5px; }
    </style>
</head>
<body>
    <div class="header">
        <h1>{$nombreMunicipio}</h1>
        <p>{$direccionMunicipio}</p>
        <p>{$comuna} - {$region}</p>
    </div>
    
    <div class="title">CERTIFICADO DE RESIDENCIA</div>
    
    <div class="info">
        <p><strong>N°:</strong> {$numeroCertificado}</p>
        <p><strong>Fecha:</strong> {$comuna}, {$fecha}</p>
    </div>
    
    <div class="content">
        <p>El que suscribe, <strong>{$cargo}</strong> de la <strong>{$nombreMunicipio}</strong>, certifica que según los antecedentes disponibles en esta oficina:</p>
    </div>
    
    <div class="data">
        <p><strong>Nombre:</strong> {$nombreCompleto}</p>
        <p><strong>RUT:</strong> {$rut}</p>
        <p><strong>Domicilio:</strong> {$direccion}, {$comuna}</p>
    </div>
    
    <div class="content">
        <p><strong>TIENE SU RESIDENCIA</strong> en el domicilio antes indicado, perteneciente a la comuna de {$comuna}.</p>
        
        <p><strong>MOTIVO DE LA SOLICITUD:</strong><br>{$motivo}</p>
        
        <p>Se extiende el presente certificado a petición del interesado para los fines que estime conveniente.</p>
    </div>
    
    <div class="footer">
        <div class="signature">
            <div class="signature-line">
                <p><strong>{$nombreUsuario}</strong></p>
                <p>{$cargo}</p>
                <p>{$nombreMunicipio}</p>
            </div>
        </div>
        
        <p style="margin-top: 40px; font-size: 9px; color: #666;">
            Certificado N° {$numeroCertificado} - Fecha de emisión: {$fecha}
        </p>
    </div>
</body>
</html>
HTML;
    }
    
    /**
     * Generar contenido para TCPDF
     */
    private function generarContenidoTCPDF($pdf, $datos, $config, $numeroCertificado) {
        // Membrete - más compacto
        $pdf->SetFont('helvetica', 'B', 14);
        $pdf->Cell(0, 8, mb_strtoupper($config['nombre_municipio']), 0, 1, 'C');
        
        $pdf->SetFont('helvetica', '', 9);
        $pdf->Cell(0, 4, $config['direccion_municipio'], 0, 1, 'C');
        $pdf->Cell(0, 4, $config['comuna'] . ' - ' . $config['region'], 0, 1, 'C');
        
        $pdf->Ln(6); // Reducido de 10 a 6
        $pdf->SetFont('helvetica', 'B', 13);
        $pdf->Cell(0, 8, 'CERTIFICADO DE RESIDENCIA', 0, 1, 'C');
        $pdf->Ln(3); // Reducido de 5 a 3
        
        $pdf->SetFont('helvetica', '', 9);
        $pdf->Cell(0, 4, 'N° ' . $numeroCertificado, 0, 1, 'R');
        $pdf->Cell(0, 4, $config['comuna'] . ', ' . $this->fechaTexto(date('Y-m-d')), 0, 1, 'R');
        $pdf->Ln(6); // Reducido de 10 a 6
        
        // Contenido principal - más compacto
        $pdf->SetFont('helvetica', '', 10);
        $usuario = $this->auth->getUser();
        $cargo = $usuario['cargo'] ?? 'Funcionario Municipal';
        
        $texto = "El que suscribe, " . $cargo . " de la " . $config['nombre_municipio'] . ", certifica que según los antecedentes disponibles:";
        $pdf->MultiCell(0, 5, $texto, 0, 'J'); // Reducido de 7 a 5
        $pdf->Ln(3);
        
        // Datos del solicitante - más compacto
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->Cell(35, 5, 'Nombre:', 0, 0); // Reducido de 40 a 35 y de 7 a 5
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(0, 5, mb_strtoupper($datos['nombre_completo']), 0, 1);
        
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->Cell(35, 5, 'RUT:', 0, 0);
        $pdf->SetFont('helvetica', '', 10);
        $pdf->Cell(0, 5, $datos['rut'], 0, 1);
        
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->Cell(35, 5, 'Domicilio:', 0, 0);
        $pdf->SetFont('helvetica', '', 10);
        $pdf->MultiCell(0, 5, mb_strtoupper($datos['direccion']) . ', ' . strtoupper($config['comuna']), 0, 'L');
        
        $pdf->Ln(3);
        
        // Declaración - más compacto
        $pdf->SetFont('helvetica', 'B', 10);
        $texto = "TIENE SU RESIDENCIA en el domicilio antes indicado, perteneciente a la comuna de " . $config['comuna'] . ".";
        $pdf->MultiCell(0, 5, $texto, 0, 'J');
        
        $pdf->Ln(2);
        
        // Motivo
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->Cell(0, 5, 'MOTIVO:', 0, 1, 'L');
        $pdf->SetFont('helvetica', '', 10);
        $pdf->MultiCell(0, 5, $datos['motivo'], 0, 'J');
        
        $pdf->Ln(2);
        
        // Cierre
        $pdf->SetFont('helvetica', '', 10);
        $pdf->MultiCell(0, 5, 'Se extiende el presente certificado a petición del interesado para los fines que estime conveniente.', 0, 'J');
        
        $pdf->Ln(8);
        
        // Firma del funcionario (tradicional) - más arriba
        $pdf->SetFont('helvetica', '', 9);
        $pdf->Cell(0, 4, '________________________________', 0, 1, 'C');
        $pdf->SetFont('helvetica', 'B', 10);
        $pdf->Cell(0, 5, mb_strtoupper($usuario['nombre_completo']), 0, 1, 'C');
        $pdf->SetFont('helvetica', '', 9);
        $pdf->Cell(0, 4, $cargo, 0, 1, 'C');
        $pdf->Cell(0, 4, $config['nombre_municipio'], 0, 1, 'C');
        
        // IMPORTANTE: NO agregar más espacio aquí para que la firma digital quepa
    }
    
    /**
     * Generar contenido de texto plano
     */
    private function generarContenidoTexto($datos, $config, $numeroCertificado) {
        $texto = strtoupper($config['nombre_municipio']) . "\\n\\n";
        $texto .= "CERTIFICADO DE RESIDENCIA\\n\\n";
        $texto .= "N: " . $numeroCertificado . "\\n\\n";
        $texto .= "Nombre: " . $datos['nombre_completo'] . "\\n";
        $texto .= "RUT: " . $datos['rut'] . "\\n";
        $texto .= "Direccion: " . $datos['direccion'] . "\\n\\n";
        $texto .= "Certifica su residencia en " . $config['comuna'];
        
        return substr($texto, 0, 200); // Limitar longitud
    }
    
    /**
     * Obtener ruta del PDF
     */
    private function getRutaPDF($solicitudId, $numeroCertificado) {
        $targetDir = CERTIFICADOS_PATH . '/' . date('Y') . '/' . date('m');
        $nombreArchivo = $numeroCertificado . '_' . $solicitudId . '.pdf';
        return $targetDir . '/' . $nombreArchivo;
    }
    
    /**
     * Firmar PDF
     */
    public function firmarPDF($certificadoId, $rutaCertificado, $passwordCertificado) {
        try {
            $sql = "SELECT * FROM certificados WHERE id = ?";
            $certificado = $this->db->selectOne($sql, [$certificadoId]);
            
            if (!$certificado) {
                throw new Exception('Certificado no encontrado');
            }
            
            $this->db->update('certificados',
                ['firmado' => 1, 'fecha_firma' => date('Y-m-d H:i:s')],
                'id = ?',
                [$certificadoId]
            );
            
            $this->auth->logActivity(
                $this->auth->getUserId(),
                $certificado['solicitud_id'],
                'certificado_firmado',
                "Certificado firmado: {$certificado['numero_certificado']}"
            );
            
            return ['success' => true, 'message' => 'Certificado firmado exitosamente'];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
    
    /**
     * Obtener configuración
     */
    private function obtenerConfiguracion() {
        $sql = "SELECT clave, valor FROM configuracion";
        $resultados = $this->db->select($sql);
        
        $config = [];
        foreach ($resultados as $row) {
            $config[$row['clave']] = $row['valor'];
        }
        
        return $config;
    }
    
    /**
     * Generar número de certificado
     */
    private function generarNumeroCertificado() {
        $prefijo = 'CERT';
        $anio = date('Y');
        
        $sql = "SELECT numero_certificado FROM certificados 
                WHERE numero_certificado LIKE ? 
                ORDER BY id DESC LIMIT 1";
        
        $ultimo = $this->db->selectOne($sql, ["{$prefijo}-{$anio}-%"]);
        
        if ($ultimo) {
            $partes = explode('-', $ultimo['numero_certificado']);
            $numero = intval($partes[2] ?? 0) + 1;
        } else {
            $numero = 1;
        }
        
        return sprintf("%s-%s-%05d", $prefijo, $anio, $numero);
    }
    
    /**
     * Convertir fecha a texto
     */
    private function fechaTexto($fecha) {
        $meses = [
            1 => 'enero', 2 => 'febrero', 3 => 'marzo', 4 => 'abril',
            5 => 'mayo', 6 => 'junio', 7 => 'julio', 8 => 'agosto',
            9 => 'septiembre', 10 => 'octubre', 11 => 'noviembre', 12 => 'diciembre'
        ];
        
        $timestamp = strtotime($fecha);
        $dia = date('d', $timestamp);
        $mes = $meses[(int)date('m', $timestamp)];
        $anio = date('Y', $timestamp);
        
        return "{$dia} de {$mes} de {$anio}";
    }
}
