<?php
/**
 * Script de prueba de login - Diagnostica problemas de autenticación
 * Ejecutar desde navegador o línea de comandos
 */

// Cargar configuración
require_once '../config/config.php';

// Deshabilitar caché
header('Cache-Control: no-cache, no-store, must-revalidate');
header('Pragma: no-cache');
header('Expires: 0');
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Test de Login - Sistema de Certificados</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
            padding: 20px;
        }
        .test-container {
            max-width: 800px;
            margin: 0 auto;
        }
        .test-result {
            padding: 15px;
            margin: 10px 0;
            border-radius: 5px;
            border-left: 4px solid;
        }
        .test-success {
            background-color: #d4edda;
            border-color: #28a745;
            color: #155724;
        }
        .test-error {
            background-color: #f8d7da;
            border-color: #dc3545;
            color: #721c24;
        }
        .test-info {
            background-color: #d1ecf1;
            border-color: #17a2b8;
            color: #0c5460;
        }
        .code-block {
            background-color: #f4f4f4;
            padding: 10px;
            border-radius: 3px;
            font-family: monospace;
            margin: 10px 0;
        }
    </style>
</head>
<body>
    <div class="test-container">
        <div class="card shadow-sm">
            <div class="card-header bg-primary text-white">
                <h3 class="mb-0">🔍 Diagnóstico de Sistema de Login</h3>
            </div>
            <div class="card-body">
                <?php
                $allTestsPassed = true;
                
                // TEST 1: Verificar PHP
                echo '<h5>Test 1: Verificación de PHP</h5>';
                $phpVersion = phpversion();
                if (version_compare($phpVersion, '8.0.0', '>=')) {
                    echo '<div class="test-result test-success">';
                    echo "✓ PHP $phpVersion - Compatible";
                    echo '</div>';
                } else {
                    echo '<div class="test-result test-error">';
                    echo "✗ PHP $phpVersion - Se requiere PHP 8.0 o superior";
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // TEST 2: Verificar funciones password
                echo '<h5 class="mt-4">Test 2: Funciones de Password</h5>';
                if (function_exists('password_hash') && function_exists('password_verify')) {
                    echo '<div class="test-result test-success">';
                    echo '✓ Funciones password_hash y password_verify disponibles';
                    echo '</div>';
                } else {
                    echo '<div class="test-result test-error">';
                    echo '✗ Funciones de password no disponibles';
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // TEST 3: Verificar hash correcto
                echo '<h5 class="mt-4">Test 3: Verificación de Hash</h5>';
                $testPassword = 'admin123';
                $correctHash = '$2y$10$eP8F5qGJxGMOZKEXvLlKQ.nLLh9s5OGLcpZPHKxQVJa7OxW6Y8KUq';
                
                if (password_verify($testPassword, $correctHash)) {
                    echo '<div class="test-result test-success">';
                    echo "✓ Hash de contraseña validado correctamente";
                    echo '<div class="code-block">';
                    echo "Contraseña: <strong>$testPassword</strong><br>";
                    echo "Hash: <small>$correctHash</small>";
                    echo '</div>';
                    echo '</div>';
                } else {
                    echo '<div class="test-result test-error">';
                    echo '✗ Error en validación de hash';
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // TEST 4: Conexión a base de datos
                echo '<h5 class="mt-4">Test 4: Conexión a Base de Datos</h5>';
                try {
                    $db = Database::getInstance();
                    echo '<div class="test-result test-success">';
                    echo '✓ Conexión a base de datos exitosa';
                    echo '<div class="code-block">';
                    echo 'Host: ' . DB_HOST . '<br>';
                    echo 'Base de datos: ' . DB_NAME . '<br>';
                    echo 'Usuario: ' . DB_USER;
                    echo '</div>';
                    echo '</div>';
                } catch (Exception $e) {
                    echo '<div class="test-result test-error">';
                    echo '✗ Error de conexión: ' . htmlspecialchars($e->getMessage());
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // TEST 5: Verificar tabla usuarios
                echo '<h5 class="mt-4">Test 5: Tabla de Usuarios</h5>';
                try {
                    $db = Database::getInstance();
                    $result = $db->query("SHOW TABLES LIKE 'usuarios'");
                    
                    if ($result->rowCount() > 0) {
                        echo '<div class="test-result test-success">';
                        echo '✓ Tabla "usuarios" existe';
                        
                        // Contar usuarios
                        $count = $db->selectOne("SELECT COUNT(*) as total FROM usuarios");
                        echo '<div class="code-block">';
                        echo 'Total de usuarios: ' . $count['total'];
                        echo '</div>';
                        echo '</div>';
                    } else {
                        echo '<div class="test-result test-error">';
                        echo '✗ Tabla "usuarios" no existe';
                        echo '<div class="code-block">';
                        echo 'Solución: Ejecutar database/schema.sql';
                        echo '</div>';
                        echo '</div>';
                        $allTestsPassed = false;
                    }
                } catch (Exception $e) {
                    echo '<div class="test-result test-error">';
                    echo '✗ Error: ' . htmlspecialchars($e->getMessage());
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // TEST 6: Verificar usuario admin
                echo '<h5 class="mt-4">Test 6: Usuario Administrador</h5>';
                try {
                    $db = Database::getInstance();
                    $admin = $db->selectOne("SELECT * FROM usuarios WHERE username = 'admin'");
                    
                    if ($admin) {
                        echo '<div class="test-result test-success">';
                        echo '✓ Usuario "admin" encontrado';
                        echo '<div class="code-block">';
                        echo 'ID: ' . $admin['id'] . '<br>';
                        echo 'Username: ' . $admin['username'] . '<br>';
                        echo 'Email: ' . $admin['email'] . '<br>';
                        echo 'Nombre: ' . $admin['nombre_completo'] . '<br>';
                        echo 'Activo: ' . ($admin['activo'] ? 'Sí' : 'No') . '<br>';
                        echo 'Longitud del hash: ' . strlen($admin['password']) . ' caracteres<br>';
                        echo 'Prefijo del hash: ' . substr($admin['password'], 0, 7);
                        echo '</div>';
                        echo '</div>';
                        
                        // TEST 7: Verificar contraseña del usuario admin
                        echo '<h5 class="mt-4">Test 7: Validación de Contraseña</h5>';
                        if (password_verify($testPassword, $admin['password'])) {
                            echo '<div class="test-result test-success">';
                            echo '<strong>✓✓✓ ÉXITO TOTAL</strong><br>';
                            echo 'La contraseña "admin123" es válida para el usuario "admin"<br>';
                            echo '<strong>El login debería funcionar correctamente</strong>';
                            echo '</div>';
                        } else {
                            echo '<div class="test-result test-error">';
                            echo '✗ La contraseña no coincide con el hash almacenado<br>';
                            echo '<strong>SOLUCIÓN:</strong> Ejecutar el script fix_password.php';
                            echo '<div class="code-block">';
                            echo 'Hash actual en BD:<br>';
                            echo '<small>' . htmlspecialchars($admin['password']) . '</small><br><br>';
                            echo 'Hash correcto a usar:<br>';
                            echo '<small>$2y$10$eP8F5qGJxGMOZKEXvLlKQ.nLLh9s5OGLcpZPHKxQVJa7OxW6Y8KUq</small>';
                            echo '</div>';
                            echo '</div>';
                            $allTestsPassed = false;
                        }
                    } else {
                        echo '<div class="test-result test-error">';
                        echo '✗ Usuario "admin" no encontrado';
                        echo '<div class="code-block">';
                        echo 'Solución: Ejecutar database/schema.sql para crear el usuario';
                        echo '</div>';
                        echo '</div>';
                        $allTestsPassed = false;
                    }
                } catch (Exception $e) {
                    echo '<div class="test-result test-error">';
                    echo '✗ Error: ' . htmlspecialchars($e->getMessage());
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // TEST 8: Verificar sesiones
                echo '<h5 class="mt-4">Test 8: Sistema de Sesiones</h5>';
                if (session_status() === PHP_SESSION_ACTIVE) {
                    echo '<div class="test-result test-success">';
                    echo '✓ Sesiones PHP activas<br>';
                    echo 'Session ID: ' . session_id();
                    echo '</div>';
                } else {
                    echo '<div class="test-result test-error">';
                    echo '✗ Sesiones no están activas';
                    echo '</div>';
                    $allTestsPassed = false;
                }
                
                // RESUMEN FINAL
                echo '<hr class="my-4">';
                echo '<h4>📊 Resumen del Diagnóstico</h4>';
                
                if ($allTestsPassed) {
                    echo '<div class="alert alert-success">';
                    echo '<h5>✅ Todos los tests pasaron correctamente</h5>';
                    echo '<p class="mb-0">El sistema de login está configurado correctamente y debería funcionar.</p>';
                    echo '<hr>';
                    echo '<h6>Credenciales de acceso:</h6>';
                    echo '<ul>';
                    echo '<li><strong>Usuario:</strong> admin</li>';
                    echo '<li><strong>Contraseña:</strong> admin123</li>';
                    echo '</ul>';
                    echo '<a href="login.php" class="btn btn-primary mt-3">';
                    echo '<i class="fas fa-sign-in-alt"></i> Ir a Login';
                    echo '</a>';
                    echo '</div>';
                } else {
                    echo '<div class="alert alert-danger">';
                    echo '<h5>❌ Algunos tests fallaron</h5>';
                    echo '<p><strong>Solución recomendada:</strong></p>';
                    echo '<ol>';
                    echo '<li>Ejecutar: <code>database/fix_password.php</code></li>';
                    echo '<li>Si el problema persiste, ejecutar: <code>database/schema.sql</code></li>';
                    echo '<li>Verificar la configuración en: <code>config/config.php</code></li>';
                    echo '</ol>';
                    echo '<a href="database/fix_password.php" class="btn btn-warning mt-3">';
                    echo 'Ejecutar Fix Password';
                    echo '</a>';
                    echo '</div>';
                }
                ?>
                
                <div class="mt-4 text-center">
                    <a href="index.php" class="btn btn-secondary">
                        Volver al Formulario
                    </a>
                    <button onclick="location.reload()" class="btn btn-info">
                        Ejecutar Tests Nuevamente
                    </button>
                </div>
            </div>
        </div>
    </div>
    
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
