<?php
/**
 * Script de diagnóstico del sistema
 * Verifica configuración, permisos y requisitos
 */

require_once '../config/config.php';

header('Content-Type: text/html; charset=UTF-8');
?>
<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="UTF-8">
    <title>Diagnóstico del Sistema</title>
    <style>
        body { font-family: monospace; padding: 20px; background: #f5f5f5; }
        .ok { color: green; font-weight: bold; }
        .error { color: red; font-weight: bold; }
        .warning { color: orange; font-weight: bold; }
        .section { background: white; padding: 15px; margin: 10px 0; border-radius: 5px; }
        h2 { border-bottom: 2px solid #333; padding-bottom: 5px; }
        pre { background: #f0f0f0; padding: 10px; border-radius: 3px; }
    </style>
</head>
<body>
    <h1>🔍 Diagnóstico del Sistema de Certificados</h1>
    
    <div class="section">
        <h2>📁 Directorios y Permisos</h2>
        <?php
        $directories = [
            'BASE_PATH' => BASE_PATH,
            'UPLOAD_PATH' => UPLOAD_PATH,
            'SOLICITUDES_PATH' => SOLICITUDES_PATH,
            'CERTIFICADOS_PATH' => CERTIFICADOS_PATH,
            'CERTIFICATES_PATH' => CERTIFICATES_PATH
        ];
        
        foreach ($directories as $name => $path) {
            echo "<strong>{$name}:</strong> {$path}<br>";
            
            if (file_exists($path)) {
                echo "  <span class='ok'>✓ Existe</span> ";
                
                if (is_dir($path)) {
                    echo "<span class='ok'>✓ Es directorio</span> ";
                    
                    if (is_writable($path)) {
                        echo "<span class='ok'>✓ Escribible</span>";
                    } else {
                        echo "<span class='error'>✗ NO escribible</span>";
                        echo "<br>  <em>Solución: chmod 755 {$path}</em>";
                    }
                } else {
                    echo "<span class='error'>✗ No es directorio</span>";
                }
            } else {
                echo "  <span class='error'>✗ No existe</span>";
                echo "<br>  <em>Solución: mkdir -p {$path} && chmod 755 {$path}</em>";
            }
            echo "<br><br>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>🔧 Extensiones PHP</h2>
        <?php
        $required_extensions = [
            'openssl' => 'Para firma digital',
            'mysqli' => 'Para base de datos',
            'gd' => 'Para manipulación de imágenes',
            'mbstring' => 'Para manejo de caracteres',
            'fileinfo' => 'Para validación de archivos'
        ];
        
        foreach ($required_extensions as $ext => $desc) {
            $loaded = extension_loaded($ext);
            echo "<strong>{$ext}</strong> ({$desc}): ";
            if ($loaded) {
                echo "<span class='ok'>✓ Instalada</span><br>";
            } else {
                echo "<span class='error'>✗ NO instalada</span><br>";
            }
        }
        ?>
    </div>
    
    <div class="section">
        <h2>📦 Composer y Librerías</h2>
        <?php
        $autoload = BASE_PATH . '/vendor/autoload.php';
        echo "<strong>Autoload de Composer:</strong> ";
        if (file_exists($autoload)) {
            echo "<span class='ok'>✓ Existe</span><br>";
            
            require_once $autoload;
            
            // Verificar TCPDF
            echo "<strong>TCPDF:</strong> ";
            if (class_exists('TCPDF')) {
                echo "<span class='ok'>✓ Disponible</span><br>";
            } else {
                echo "<span class='error'>✗ No disponible</span><br>";
                echo "<em>Solución: composer require tecnickcom/tcpdf</em><br>";
            }
            
            // Verificar Dompdf
            echo "<strong>Dompdf:</strong> ";
            if (class_exists('\Dompdf\Dompdf')) {
                echo "<span class='ok'>✓ Disponible</span><br>";
            } else {
                echo "<span class='warning'>⚠ No disponible (opcional)</span><br>";
            }
            
        } else {
            echo "<span class='error'>✗ No existe</span><br>";
            echo "<em>Solución: composer install</em><br>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>📊 Configuración PHP</h2>
        <?php
        $configs = [
            'upload_max_filesize' => ini_get('upload_max_filesize'),
            'post_max_size' => ini_get('post_max_size'),
            'max_execution_time' => ini_get('max_execution_time'),
            'memory_limit' => ini_get('memory_limit'),
            'file_uploads' => ini_get('file_uploads') ? 'Habilitado' : 'Deshabilitado'
        ];
        
        foreach ($configs as $key => $value) {
            echo "<strong>{$key}:</strong> {$value}<br>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>🗄️ Base de Datos</h2>
        <?php
        try {
            $db = Database::getInstance();
            echo "<span class='ok'>✓ Conexión exitosa</span><br>";
            
            // Verificar tablas
            $tables = ['usuarios', 'solicitudes', 'certificados', 'logs_actividad'];
            echo "<br><strong>Tablas:</strong><br>";
            
            foreach ($tables as $table) {
                $result = $db->select("SHOW TABLES LIKE '{$table}'");
                if (count($result) > 0) {
                    echo "  {$table}: <span class='ok'>✓ Existe</span><br>";
                } else {
                    echo "  {$table}: <span class='error'>✗ No existe</span><br>";
                }
            }
            
        } catch (Exception $e) {
            echo "<span class='error'>✗ Error de conexión</span><br>";
            echo "Error: " . $e->getMessage() . "<br>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>👤 Usuario de Prueba</h2>
        <?php
        try {
            $db = Database::getInstance();
            $user = $db->selectOne("SELECT * FROM usuarios WHERE email = 'admin'");
            
            if ($user) {
                echo "<strong>Usuario admin:</strong> <span class='ok'>✓ Existe</span><br>";
                echo "ID: {$user['id']}<br>";
                echo "Nombre: {$user['nombre_completo']}<br>";
                echo "Email: {$user['email']}<br>";
            } else {
                echo "<span class='error'>✗ Usuario admin no existe</span><br>";
                echo "<em>Solución: Ejecutar database/schema.sql</em><br>";
            }
        } catch (Exception $e) {
            echo "<span class='error'>Error: " . $e->getMessage() . "</span><br>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>🧪 Test de Generación Básica</h2>
        <?php
        try {
            // Verificar que se pueda crear instancia
            $cert = new CertificadoPDF();
            echo "<strong>CertificadoPDF:</strong> <span class='ok'>✓ Clase cargada</span><br>";
            
            // Verificar auth
            $auth = new Auth();
            echo "<strong>Auth:</strong> <span class='ok'>✓ Clase cargada</span><br>";
            
        } catch (Exception $e) {
            echo "<span class='error'>Error: " . $e->getMessage() . "</span><br>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>📝 Logs Recientes</h2>
        <?php
        $logFile = BASE_PATH . '/logs/app.log';
        if (file_exists($logFile)) {
            echo "<strong>Archivo de logs:</strong> {$logFile}<br>";
            $lines = file($logFile);
            $recent = array_slice($lines, -20);
            echo "<pre>";
            echo htmlspecialchars(implode('', $recent));
            echo "</pre>";
        } else {
            echo "<span class='warning'>⚠ No hay archivo de logs</span><br>";
        }
        
        // PHP error log
        $phpErrorLog = ini_get('error_log');
        if ($phpErrorLog && file_exists($phpErrorLog)) {
            echo "<br><strong>PHP Error Log:</strong> {$phpErrorLog}<br>";
            $lines = file($phpErrorLog);
            $recent = array_slice($lines, -10);
            echo "<pre>";
            echo htmlspecialchars(implode('', $recent));
            echo "</pre>";
        }
        ?>
    </div>
    
    <div class="section">
        <h2>✅ Resumen</h2>
        <p>Si todos los checks anteriores son <span class="ok">✓</span>, el sistema debería funcionar correctamente.</p>
        <p>Si hay errores <span class="error">✗</span>, sigue las soluciones sugeridas.</p>
        <br>
        <a href="admin/dashboard.php">← Volver al Dashboard</a>
    </div>
</body>
</html>
