<?php
/**
 * API para manejo de certificados
 */

require_once '../../config/config.php';

header('Content-Type: application/json');

$auth = new Auth();
$auth->requireLogin();

try {
    $method = $_SERVER['REQUEST_METHOD'];
    $action = $_GET['action'] ?? '';
    
    $certificadoPDF = new CertificadoPDF();
    $mailer = new Mailer();
    
    switch ($action) {
        case 'generar':
            handleGenerar($certificadoPDF);
            break;
            
        case 'firmar':
            handleFirmar($certificadoPDF);
            break;
            
        case 'enviar':
            handleEnviar($mailer);
            break;
            
        case 'descargar':
            handleDescargar();
            break;
            
        default:
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'Acción no válida'
            ]);
            break;
    }
    
} catch (Exception $e) {
    error_log("Error en API certificados: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error interno del servidor: ' . $e->getMessage()
    ]);
}

/**
 * Generar certificado PDF (con opción de firma automática)
 */
function handleGenerar($certificadoPDF) {
    try {
        if (empty($_POST['solicitud_id'])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => 'ID de solicitud requerido'
            ]);
            return;
        }
        
        $solicitudId = intval($_POST['solicitud_id']);
        
        // Verificar si se envió certificado digital para firma automática
        $opcionesFirma = null;
        
        if (isset($_FILES['certificado_pfx']) && isset($_POST['password_pfx'])) {
            // Procesar archivo del certificado digital
            $certificadoPfxFile = $_FILES['certificado_pfx'];
            
            // Log para debug
            error_log("=== GENERANDO CON FIRMA ===");
            error_log("Archivo recibido: " . $certificadoPfxFile['name']);
            error_log("Error code: " . $certificadoPfxFile['error']);
            error_log("Tamaño: " . $certificadoPfxFile['size']);
            
            if ($certificadoPfxFile['error'] !== UPLOAD_ERR_OK) {
                $errorMsg = 'Error al cargar el archivo';
                switch ($certificadoPfxFile['error']) {
                    case UPLOAD_ERR_INI_SIZE:
                    case UPLOAD_ERR_FORM_SIZE:
                        $errorMsg = 'El archivo es demasiado grande';
                        break;
                    case UPLOAD_ERR_PARTIAL:
                        $errorMsg = 'El archivo se cargó parcialmente';
                        break;
                    case UPLOAD_ERR_NO_FILE:
                        $errorMsg = 'No se seleccionó ningún archivo';
                        break;
                    default:
                        $errorMsg = 'Error desconocido al cargar el archivo (código: ' . $certificadoPfxFile['error'] . ')';
                }
                
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => $errorMsg
                ]);
                return;
            }
            
            // Validar extensión
            $extension = strtolower(pathinfo($certificadoPfxFile['name'], PATHINFO_EXTENSION));
            if (!in_array($extension, ['pfx', 'p12'])) {
                http_response_code(400);
                echo json_encode([
                    'success' => false,
                    'message' => 'El certificado digital debe ser .pfx o .p12'
                ]);
                return;
            }
            
            // Mover archivo temporal a ubicación segura
            $uploadDir = CERTIFICATES_PATH;
            if (!is_dir($uploadDir)) {
                if (!mkdir($uploadDir, 0755, true)) {
                    error_log("ERROR: No se pudo crear directorio: " . $uploadDir);
                    http_response_code(500);
                    echo json_encode([
                        'success' => false,
                        'message' => 'No se pudo crear directorio para certificados'
                    ]);
                    return;
                }
            }
            
            $nombreArchivo = 'temp_cert_' . uniqid() . '.' . $extension;
            $rutaCertificado = $uploadDir . '/' . $nombreArchivo;
            
            error_log("Moviendo archivo a: " . $rutaCertificado);
            
            if (!move_uploaded_file($certificadoPfxFile['tmp_name'], $rutaCertificado)) {
                error_log("ERROR: No se pudo mover archivo temporal");
                http_response_code(500);
                echo json_encode([
                    'success' => false,
                    'message' => 'No se pudo guardar el certificado digital temporalmente'
                ]);
                return;
            }
            
            error_log("Archivo movido correctamente");
            
            $opcionesFirma = [
                'certificado_pfx' => $rutaCertificado,
                'password' => $_POST['password_pfx']
            ];
        }
        
        // Generar certificado (con o sin firma)
        error_log("Llamando a generar certificado...");
        $resultado = $certificadoPDF->generar($solicitudId, $opcionesFirma);
        error_log("Resultado: " . print_r($resultado, true));
        
        // Limpiar archivo temporal del certificado
        if ($opcionesFirma && file_exists($opcionesFirma['certificado_pfx'])) {
            @unlink($opcionesFirma['certificado_pfx']);
            error_log("Archivo temporal eliminado");
        }
        
        if ($resultado['success']) {
            http_response_code(201);
        } else {
            http_response_code(400);
        }
        
        echo json_encode($resultado);
        
    } catch (Exception $e) {
        error_log("=== ERROR EN handleGenerar ===");
        error_log("Mensaje: " . $e->getMessage());
        error_log("Archivo: " . $e->getFile() . " Línea: " . $e->getLine());
        error_log("Stack: " . $e->getTraceAsString());
        
        http_response_code(500);
        echo json_encode([
            'success' => false,
            'message' => 'Error interno: ' . $e->getMessage()
        ]);
    }
}

/**
 * Firmar certificado digitalmente
 */
function handleFirmar($certificadoPDF) {
    if (empty($_POST['certificado_id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'ID de certificado requerido'
        ]);
        return;
    }
    
    // Verificar si se subió un certificado digital
    if (empty($_FILES['certificado_pfx'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Archivo de certificado digital requerido'
        ]);
        return;
    }
    
    if (empty($_POST['password_pfx'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'Contraseña del certificado requerida'
        ]);
        return;
    }
    
    $certificadoId = intval($_POST['certificado_id']);
    $passwordPfx = $_POST['password_pfx'];
    
    // Guardar certificado temporal
    $tmpFile = tempnam(sys_get_temp_dir(), 'cert_');
    move_uploaded_file($_FILES['certificado_pfx']['tmp_name'], $tmpFile);
    
    // Firmar PDF
    $resultado = $certificadoPDF->firmarPDF($certificadoId, $tmpFile, $passwordPfx);
    
    // Eliminar archivo temporal
    unlink($tmpFile);
    
    if ($resultado['success']) {
        http_response_code(200);
    } else {
        http_response_code(400);
    }
    
    echo json_encode($resultado);
}

/**
 * Enviar certificado por correo
 */
function handleEnviar($mailer) {
    if (empty($_POST['certificado_id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'ID de certificado requerido'
        ]);
        return;
    }
    
    $certificadoId = intval($_POST['certificado_id']);
    $resultado = $mailer->enviarCertificado($certificadoId);
    
    if ($resultado['success']) {
        http_response_code(200);
    } else {
        http_response_code(400);
    }
    
    echo json_encode($resultado);
}

/**
 * Descargar certificado PDF
 */
function handleDescargar() {
    if (empty($_GET['certificado_id'])) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'ID de certificado requerido'
        ]);
        return;
    }
    
    $db = Database::getInstance();
    $certificadoId = intval($_GET['certificado_id']);
    
    $sql = "SELECT c.*, s.nombre_completo 
            FROM certificados c
            INNER JOIN solicitudes s ON c.solicitud_id = s.id
            WHERE c.id = ?";
    
    $certificado = $db->selectOne($sql, [$certificadoId]);
    
    if (!$certificado) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Certificado no encontrado'
        ]);
        return;
    }
    
    if (!file_exists($certificado['ruta_pdf'])) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Archivo PDF no encontrado'
        ]);
        return;
    }
    
    // Registrar descarga
    $auth = new Auth();
    $auth->logActivity(
        $auth->getUserId(),
        $certificado['solicitud_id'],
        'certificado_descargado',
        "Certificado descargado: {$certificado['numero_certificado']}"
    );
    
    // Enviar archivo
    header('Content-Type: application/pdf');
    header('Content-Disposition: attachment; filename="Certificado_' . $certificado['numero_certificado'] . '.pdf"');
    header('Content-Length: ' . filesize($certificado['ruta_pdf']));
    header('Cache-Control: private, max-age=0, must-revalidate');
    header('Pragma: public');
    
    readfile($certificado['ruta_pdf']);
    exit;
}
