/**
 * JavaScript para formulario de solicitud de certificados
 */

$(document).ready(function() {
    let selectedFiles = [];
    const maxFileSize = 5 * 1024 * 1024; // 5MB
    const maxFiles = 5;
    const allowedExtensions = ['pdf', 'jpg', 'jpeg', 'png'];

    // Formatear RUT automáticamente
    $('#rutInput').on('input', function() {
        let rut = $(this).val().replace(/[^0-9kK]/g, '');
        
        if (rut.length > 1) {
            let dv = rut.slice(-1);
            let numero = rut.slice(0, -1);
            
            // Formatear con puntos
            numero = numero.replace(/\B(?=(\d{3})+(?!\d))/g, '.');
            
            $(this).val(numero + '-' + dv);
        }
    });

    // Validar RUT
    function validarRut(rut) {
        rut = rut.replace(/[^0-9kK]/g, '');
        if (rut.length < 2) return false;
        
        let dv = rut.slice(-1).toUpperCase();
        let numero = rut.slice(0, -1);
        
        let suma = 0;
        let multiplo = 2;
        
        for (let i = numero.length - 1; i >= 0; i--) {
            suma += parseInt(numero[i]) * multiplo;
            multiplo = multiplo < 7 ? multiplo + 1 : 2;
        }
        
        let dvEsperado = 11 - (suma % 11);
        dvEsperado = dvEsperado === 11 ? '0' : (dvEsperado === 10 ? 'K' : dvEsperado.toString());
        
        return dv === dvEsperado;
    }

    // Manejo de archivos
    $('#fileInput').on('change', function(e) {
        const files = Array.from(e.target.files);
        
        if (selectedFiles.length + files.length > maxFiles) {
            showAlert('danger', `Solo puede adjuntar un máximo de ${maxFiles} archivos.`);
            return;
        }
        
        files.forEach(file => {
            // Validar tamaño
            if (file.size > maxFileSize) {
                showAlert('danger', `El archivo ${file.name} excede el tamaño máximo de 5MB.`);
                return;
            }
            
            // Validar extensión
            const extension = file.name.split('.').pop().toLowerCase();
            if (!allowedExtensions.includes(extension)) {
                showAlert('danger', `El archivo ${file.name} tiene un formato no permitido.`);
                return;
            }
            
            selectedFiles.push(file);
        });
        
        updateFileList();
        $(this).val(''); // Limpiar input
    });

    // Actualizar lista de archivos
    function updateFileList() {
        const fileList = $('#fileList');
        fileList.empty();
        
        selectedFiles.forEach((file, index) => {
            const fileSize = formatFileSize(file.size);
            const fileItem = $(`
                <div class="file-item">
                    <i class="fas fa-file-${getFileIcon(file.name)} me-2"></i>
                    <span class="file-name">${file.name}</span>
                    <span class="file-size">${fileSize}</span>
                    <button type="button" class="btn-remove-file" data-index="${index}">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `);
            
            fileList.append(fileItem);
        });
    }

    // Eliminar archivo
    $(document).on('click', '.btn-remove-file', function() {
        const index = $(this).data('index');
        selectedFiles.splice(index, 1);
        updateFileList();
    });

    // Formatear tamaño de archivo
    function formatFileSize(bytes) {
        if (bytes === 0) return '0 Bytes';
        const k = 1024;
        const sizes = ['Bytes', 'KB', 'MB'];
        const i = Math.floor(Math.log(bytes) / Math.log(k));
        return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
    }

    // Obtener icono de archivo
    function getFileIcon(filename) {
        const extension = filename.split('.').pop().toLowerCase();
        switch(extension) {
            case 'pdf': return 'pdf';
            case 'jpg':
            case 'jpeg':
            case 'png': return 'image';
            default: return 'alt';
        }
    }

    // Submit del formulario
    $('#solicitudForm').on('submit', function(e) {
        e.preventDefault();
        
        // Validar RUT
        const rut = $('#rutInput').val();
        if (!validarRut(rut)) {
            showAlert('danger', 'El RUT ingresado no es válido.');
            $('#rutInput').focus();
            return;
        }
        
        // Validar términos
        if (!$('#terminos').is(':checked')) {
            showAlert('danger', 'Debe aceptar los términos y condiciones.');
            return;
        }
        
        // Validar reCAPTCHA (en producción)
        // const recaptchaResponse = grecaptcha.getResponse();
        // if (!recaptchaResponse) {
        //     showAlert('danger', 'Por favor, complete el reCAPTCHA.');
        //     return;
        // }
        
        // Preparar FormData
        const formData = new FormData(this);
        
        // Eliminar archivos del formulario y agregar los seleccionados
        formData.delete('adjuntos[]');
        selectedFiles.forEach(file => {
            formData.append('adjuntos[]', file);
        });
        
        // Mostrar loading
        $('#loadingOverlay').addClass('active');
        
        // Enviar solicitud
        $.ajax({
            url: 'api/solicitudes.php',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            success: function(response) {
                $('#loadingOverlay').removeClass('active');
                
                if (response.success) {
                    // Mostrar mensaje de éxito
                    $('#formContainer').hide();
                    $('#numeroSolicitud').text(response.numero_solicitud);
                    $('#successMessage').fadeIn();
                    
                    // Scroll al inicio
                    $('html, body').animate({ scrollTop: 0 }, 500);
                } else {
                    showAlert('danger', response.message || 'Error al procesar la solicitud.');
                }
            },
            error: function(xhr) {
                $('#loadingOverlay').removeClass('active');
                
                let errorMessage = 'Error al enviar la solicitud. Por favor, intente nuevamente.';
                
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    errorMessage = xhr.responseJSON.message;
                }
                
                showAlert('danger', errorMessage);
            }
        });
    });

    // Mostrar alertas
    function showAlert(type, message) {
        const alertDiv = $(`
            <div class="alert alert-${type} alert-dismissible fade show" role="alert">
                <i class="fas fa-${type === 'success' ? 'check-circle' : 'exclamation-circle'} me-2"></i>
                ${message}
                <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
            </div>
        `);
        
        $('.form-card').prepend(alertDiv);
        
        // Scroll a la alerta
        $('html, body').animate({
            scrollTop: alertDiv.offset().top - 100
        }, 500);
        
        // Auto-cerrar después de 5 segundos
        setTimeout(function() {
            alertDiv.alert('close');
        }, 5000);
    }

    // Prevenir doble submit
    let isSubmitting = false;
    $('#solicitudForm').on('submit', function(e) {
        if (isSubmitting) {
            e.preventDefault();
            return false;
        }
        isSubmitting = true;
        
        setTimeout(function() {
            isSubmitting = false;
        }, 3000);
    });
});
