/**
 * JavaScript principal para Panel de Administración
 */

$(document).ready(function() {
    // Inicializar tooltips
    var tooltipTriggerList = [].slice.call(document.querySelectorAll('[data-bs-toggle="tooltip"]'));
    var tooltipList = tooltipTriggerList.map(function (tooltipTriggerEl) {
        return new bootstrap.Tooltip(tooltipTriggerEl);
    });

    // Auto-cerrar alertas
    $('.alert').not('.alert-permanent').delay(5000).fadeOut(350);

    // Confirmación antes de eliminar
    $('.btn-delete').on('click', function(e) {
        if (!confirm('¿Está seguro de que desea eliminar este elemento?')) {
            e.preventDefault();
            return false;
        }
    });

    // Marcar enlace activo en sidebar
    markActiveMenuItem();
});

/**
 * Marcar elemento activo en el menú
 */
function markActiveMenuItem() {
    const currentPath = window.location.pathname;
    
    $('.sidebar .nav-link').each(function() {
        const href = $(this).attr('href');
        
        if (currentPath.includes(href)) {
            $(this).addClass('active');
        }
    });
}

/**
 * Mostrar loading overlay
 */
function showLoading(message = 'Procesando...') {
    const overlay = $('<div class="loading-overlay"></div>');
    const content = $(`
        <div class="text-center">
            <div class="spinner-border text-light" style="width: 3rem; height: 3rem;" role="status">
                <span class="visually-hidden">Cargando...</span>
            </div>
            <p class="text-white mt-3">${message}</p>
        </div>
    `);
    
    overlay.append(content);
    $('body').append(overlay);
}

/**
 * Ocultar loading overlay
 */
function hideLoading() {
    $('.loading-overlay').remove();
}

/**
 * Mostrar notificación toast
 */
function showToast(type, message) {
    const bgClass = {
        'success': 'bg-success',
        'error': 'bg-danger',
        'warning': 'bg-warning',
        'info': 'bg-info'
    }[type] || 'bg-info';

    const icon = {
        'success': 'fa-check-circle',
        'error': 'fa-exclamation-circle',
        'warning': 'fa-exclamation-triangle',
        'info': 'fa-info-circle'
    }[type] || 'fa-info-circle';

    const toast = $(`
        <div class="toast align-items-center text-white ${bgClass} border-0" role="alert" aria-live="assertive" aria-atomic="true">
            <div class="d-flex">
                <div class="toast-body">
                    <i class="fas ${icon} me-2"></i>${message}
                </div>
                <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
            </div>
        </div>
    `);

    // Contenedor de toasts
    if ($('.toast-container').length === 0) {
        $('body').append('<div class="toast-container position-fixed top-0 end-0 p-3" style="z-index: 9999;"></div>');
    }

    $('.toast-container').append(toast);
    
    const bsToast = new bootstrap.Toast(toast[0], {
        autohide: true,
        delay: 5000
    });
    
    bsToast.show();
    
    toast.on('hidden.bs.toast', function() {
        $(this).remove();
    });
}

/**
 * Formatear fecha
 */
function formatDate(dateString) {
    const date = new Date(dateString);
    const day = String(date.getDate()).padStart(2, '0');
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const year = date.getFullYear();
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    
    return `${day}/${month}/${year} ${hours}:${minutes}`;
}

/**
 * Formatear tamaño de archivo
 */
function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return Math.round(bytes / Math.pow(k, i) * 100) / 100 + ' ' + sizes[i];
}

/**
 * Obtener badge de estado
 */
function getEstadoBadge(estado) {
    const badges = {
        'pendiente': '<span class="badge bg-warning text-dark"><i class="fas fa-clock me-1"></i>Pendiente</span>',
        'en_proceso': '<span class="badge bg-info"><i class="fas fa-spinner me-1"></i>En Proceso</span>',
        'aprobada': '<span class="badge bg-success"><i class="fas fa-check-circle me-1"></i>Aprobada</span>',
        'rechazada': '<span class="badge bg-danger"><i class="fas fa-times-circle me-1"></i>Rechazada</span>',
        'enviada': '<span class="badge bg-primary"><i class="fas fa-paper-plane me-1"></i>Enviada</span>'
    };
    
    return badges[estado] || '<span class="badge bg-secondary">Desconocido</span>';
}

/**
 * Actualizar estado de solicitud
 */
function actualizarEstado(solicitudId, nuevoEstado, observaciones = null) {
    showLoading('Actualizando estado...');
    
    $.ajax({
        url: '../api/solicitudes.php',
        type: 'PUT',
        data: {
            id: solicitudId,
            estado: nuevoEstado,
            observaciones: observaciones
        },
        success: function(response) {
            hideLoading();
            
            if (response.success) {
                showToast('success', 'Estado actualizado exitosamente');
                setTimeout(() => location.reload(), 1000);
            } else {
                showToast('error', response.message || 'Error al actualizar el estado');
            }
        },
        error: function(xhr) {
            hideLoading();
            showToast('error', 'Error al comunicarse con el servidor');
        }
    });
}

/**
 * Generar certificado (con opción de firma)
 */
function generarCertificado(solicitudId) {
    // Mostrar modal con opciones
    mostrarModalGenerarCertificado(solicitudId);
}

/**
 * Mostrar modal para generar certificado con opción de firma
 */
function mostrarModalGenerarCertificado(solicitudId) {
    const modalHtml = `
        <div class="modal fade" id="modalGenerarCertificado" tabindex="-1">
            <div class="modal-dialog">
                <div class="modal-content">
                    <div class="modal-header bg-success text-white">
                        <h5 class="modal-title">
                            <i class="fas fa-file-pdf me-2"></i>Generar Certificado
                        </h5>
                        <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                    </div>
                    <form id="formGenerarCertificado" enctype="multipart/form-data">
                        <div class="modal-body">
                            <input type="hidden" name="solicitud_id" value="${solicitudId}">
                            
                            <div class="alert alert-info">
                                <i class="fas fa-info-circle me-2"></i>
                                <strong>¿Desea firmar el certificado automáticamente?</strong>
                                <p class="mb-0 small mt-2">Si desea generar y firmar en un solo paso, active la opción y cargue su certificado digital.</p>
                            </div>
                            
                            <div class="form-check form-switch mb-3">
                                <input class="form-check-input" type="checkbox" id="checkFirmarAuto">
                                <label class="form-check-label" for="checkFirmarAuto">
                                    <strong>Generar y firmar automáticamente</strong>
                                </label>
                            </div>
                            
                            <div id="opcionesFirma" style="display: none;">
                                <hr>
                                <h6><i class="fas fa-signature me-2"></i>Datos de Firma Digital</h6>
                                
                                <div class="mb-3">
                                    <label class="form-label">Certificado Digital (.pfx, .p12):</label>
                                    <input type="file" class="form-control" id="certificado_pfx" name="certificado_pfx" accept=".pfx,.p12">
                                    <small class="text-muted">Seleccione su certificado digital</small>
                                </div>
                                
                                <div class="mb-3">
                                    <label class="form-label">Contraseña del Certificado:</label>
                                    <div class="input-group">
                                        <input type="password" class="form-control" id="password_pfx" name="password_pfx" placeholder="Contraseña del certificado">
                                        <button class="btn btn-outline-secondary" type="button" onclick="togglePasswordGenerar()">
                                            <i class="fas fa-eye" id="iconPasswordGenerar"></i>
                                        </button>
                                    </div>
                                </div>
                                
                                <div class="alert alert-warning small">
                                    <i class="fas fa-exclamation-triangle me-2"></i>
                                    <strong>Importante:</strong> La firma se agregará en la esquina inferior izquierda del certificado.
                                </div>
                            </div>
                        </div>
                        <div class="modal-footer">
                            <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancelar</button>
                            <button type="submit" class="btn btn-success">
                                <i class="fas fa-file-pdf me-2"></i>
                                <span id="textoBotonGenerar">Generar Certificado</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </div>
    `;
    
    // Remover modal existente si hay
    $('#modalGenerarCertificado').remove();
    
    // Agregar modal al body
    $('body').append(modalHtml);
    
    // Mostrar modal
    const modal = new bootstrap.Modal(document.getElementById('modalGenerarCertificado'));
    modal.show();
    
    // Event listener para checkbox de firma automática
    $('#checkFirmarAuto').on('change', function() {
        if (this.checked) {
            $('#opcionesFirma').slideDown();
            $('#textoBotonGenerar').text('Generar y Firmar Certificado');
            $('#certificado_pfx').prop('required', true);
            $('#password_pfx').prop('required', true);
        } else {
            $('#opcionesFirma').slideUp();
            $('#textoBotonGenerar').text('Generar Certificado');
            $('#certificado_pfx').prop('required', false);
            $('#password_pfx').prop('required', false);
        }
    });
    
    // Manejar envío del formulario
    $('#formGenerarCertificado').on('submit', function(e) {
        e.preventDefault();
        
        const formData = new FormData(this);
        const conFirma = $('#checkFirmarAuto').is(':checked');
        
        showLoading(conFirma ? 'Generando y firmando certificado...' : 'Generando certificado...');
        
        $.ajax({
            url: '../api/certificados.php?action=generar',
            type: 'POST',
            data: formData,
            processData: false,
            contentType: false,
            dataType: 'json',
            success: function(response) {
                hideLoading();
                
                if (response.success) {
                    let mensaje = response.message;
                    if (response.firmado && response.firmante) {
                        mensaje += `<br><small>Firmado por: ${response.firmante}</small>`;
                    }
                    showToast('success', mensaje);
                    
                    // Cerrar modal
                    modal.hide();
                    
                    // Recargar página después de 1.5 segundos
                    setTimeout(() => location.reload(), 1500);
                } else {
                    showToast('error', response.message);
                }
            },
            error: function(xhr, status, error) {
                hideLoading();
                
                console.error('Error al generar certificado:', {
                    status: xhr.status,
                    responseText: xhr.responseText,
                    error: error
                });
                
                let message = 'Error al comunicarse con el servidor';
                
                if (xhr.responseJSON && xhr.responseJSON.message) {
                    message = xhr.responseJSON.message;
                } else if (xhr.status === 404) {
                    message = 'API no encontrada';
                } else if (xhr.status === 500) {
                    message = 'Error interno del servidor';
                }
                
                showToast('error', message);
            }
        });
    });
}

/**
 * Toggle password visibility en modal de generar
 */
function togglePasswordGenerar() {
    const input = $('#password_pfx');
    const icon = $('#iconPasswordGenerar');
    
    if (input.attr('type') === 'password') {
        input.attr('type', 'text');
        icon.removeClass('fa-eye').addClass('fa-eye-slash');
    } else {
        input.attr('type', 'password');
        icon.removeClass('fa-eye-slash').addClass('fa-eye');
    }
}

/**
 * Enviar certificado por correo
 */
function enviarCertificado(certificadoId) {
    if (!confirm('¿Desea enviar el certificado por correo electrónico?')) {
        return;
    }
    
    showLoading('Enviando certificado...');
    
    $.ajax({
        url: '../api/certificados.php?action=enviar',
        type: 'POST',
        data: {
            certificado_id: certificadoId
        },
        success: function(response) {
            hideLoading();
            
            if (response.success) {
                showToast('success', 'Certificado enviado exitosamente');
                setTimeout(() => location.reload(), 1000);
            } else {
                showToast('error', response.message || 'Error al enviar el certificado');
            }
        },
        error: function(xhr) {
            hideLoading();
            const message = xhr.responseJSON?.message || 'Error al comunicarse con el servidor';
            showToast('error', message);
        }
    });
}

/**
 * Descargar certificado
 */
function descargarCertificado(certificadoId) {
    window.location.href = `../api/certificados.php?action=descargar&certificado_id=${certificadoId}`;
}

/**
 * Validar formulario antes de enviar
 */
function validarFormulario(formId) {
    const form = document.getElementById(formId);
    
    if (!form.checkValidity()) {
        form.classList.add('was-validated');
        return false;
    }
    
    return true;
}

/**
 * Copiar al portapapeles
 */
function copiarAlPortapapeles(texto) {
    navigator.clipboard.writeText(texto).then(() => {
        showToast('success', 'Copiado al portapapeles');
    }).catch(() => {
        showToast('error', 'Error al copiar');
    });
}

/**
 * Confirmar acción
 */
function confirmarAccion(mensaje, callback) {
    if (confirm(mensaje)) {
        callback();
    }
}

/**
 * Exportar tabla a Excel
 */
function exportarAExcel(tableId, filename) {
    const table = document.getElementById(tableId);
    const wb = XLSX.utils.table_to_book(table);
    XLSX.writeFile(wb, filename + '.xlsx');
}

/**
 * Imprimir elemento
 */
function imprimirElemento(elementId) {
    const printContent = document.getElementById(elementId);
    const windowUrl = 'about:blank';
    const uniqueName = new Date();
    const windowName = 'Print' + uniqueName.getTime();
    
    const printWindow = window.open(windowUrl, windowName, 'width=800,height=600');
    
    printWindow.document.write(`
        <!DOCTYPE html>
        <html>
        <head>
            <title>Imprimir</title>
            <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet">
            <style>
                @media print {
                    .no-print { display: none !important; }
                }
            </style>
        </head>
        <body>
            ${printContent.innerHTML}
        </body>
        </html>
    `);
    
    printWindow.document.close();
    printWindow.focus();
    
    setTimeout(() => {
        printWindow.print();
        printWindow.close();
    }, 250);
}
