# 🔐 Guía de Solución de Problemas de Login

## ⚠️ Problema: "Usuario o contraseña incorrectos"

Esta guía te ayudará a resolver definitivamente cualquier problema de acceso al panel de administración.

---

## ✅ SOLUCIÓN RÁPIDA (Recomendada)

### Método 1: Ejecutar el Script Fix Password

1. **Acceder al servidor/hosting**

2. **Ejecutar el script desde navegador:**
   ```
   http://tu-dominio.com/certificados-residencia/database/fix_password.php
   ```

   O desde línea de comandos:
   ```bash
   cd /ruta/certificados-residencia/database
   php fix_password.php
   ```

3. **Verás un mensaje como:**
   ```
   Hash generado: $2y$10$eP8F5qGJxGMOZKEXvLlKQ...
   Contraseña: admin123
   
   ✓ Contraseña actualizada exitosamente
   Usuario: admin
   Contraseña: admin123
   
   ✓✓✓ VERIFICACIÓN EXITOSA - El login funcionará correctamente
   ```

4. **Probar el login:**
   - Usuario: `admin`
   - Contraseña: `admin123`

---

## 🔧 SOLUCIÓN MANUAL (Alternativa)

### Método 2: Actualizar Directamente en MySQL

1. **Acceder a MySQL:**
   ```bash
   mysql -u root -p
   ```

2. **Seleccionar la base de datos:**
   ```sql
   USE certificados_residencia;
   ```

3. **Actualizar la contraseña con el hash correcto:**
   ```sql
   UPDATE usuarios 
   SET password = '$2y$10$eP8F5qGJxGMOZKEXvLlKQ.nLLh9s5OGLcpZPHKxQVJa7OxW6Y8KUq' 
   WHERE username = 'admin';
   ```

4. **Verificar la actualización:**
   ```sql
   SELECT username, email FROM usuarios WHERE username = 'admin';
   ```

5. **Probar el login:**
   - Usuario: `admin`
   - Contraseña: `admin123`

---

## 🔑 GENERAR NUEVA CONTRASEÑA PERSONALIZADA

Si quieres usar una contraseña diferente:

### Opción A: Usando PHP en línea de comandos

```bash
php -r "echo password_hash('tu_contraseña_aqui', PASSWORD_BCRYPT) . PHP_EOL;"
```

Esto generará un hash como:
```
$2y$10$AbCdEf...
```

### Opción B: Usando un script PHP

Crear archivo `generar_hash.php`:

```php
<?php
$password = 'tu_nueva_contraseña';
$hash = password_hash($password, PASSWORD_BCRYPT);
echo "Hash generado:\n$hash\n\n";
echo "Ejecuta en MySQL:\n";
echo "UPDATE usuarios SET password = '$hash' WHERE username = 'admin';\n";
?>
```

Ejecutar:
```bash
php generar_hash.php
```

### Opción C: Usando phpMyAdmin

1. Acceder a phpMyAdmin
2. Seleccionar base de datos `certificados_residencia`
3. Tabla `usuarios`
4. Editar usuario `admin`
5. En el campo `password`, usar la función `PASSWORD()` o `MD5()` no funciona con bcrypt
6. **Mejor usar los métodos A o B**

---

## 🔍 DIAGNÓSTICO DE PROBLEMAS

### Verificar que la tabla usuarios existe:

```sql
USE certificados_residencia;
SHOW TABLES LIKE 'usuarios';
```

### Verificar que el usuario admin existe:

```sql
SELECT * FROM usuarios WHERE username = 'admin';
```

Deberías ver:
- id: 1
- username: admin
- password: (un hash largo)
- nombre_completo: Administrador del Sistema
- email: admin@municipio.cl

### Verificar el hash de la contraseña:

```sql
SELECT username, LENGTH(password) as hash_length, SUBSTRING(password, 1, 7) as hash_prefix 
FROM usuarios WHERE username = 'admin';
```

Resultado esperado:
- hash_length: 60
- hash_prefix: $2y$10$

---

## 📝 CREDENCIALES POR DEFECTO

Después de ejecutar cualquiera de las soluciones anteriores:

```
Usuario: admin
Contraseña: admin123
```

---

## 🚨 PROBLEMAS COMUNES Y SOLUCIONES

### Problema 1: "Base de datos no encontrada"

**Solución:**
```bash
mysql -u root -p < database/schema.sql
```

### Problema 2: "Tabla usuarios no existe"

**Solución:**
```sql
USE certificados_residencia;
SOURCE /ruta/completa/database/schema.sql;
```

### Problema 3: "La sesión expira inmediatamente"

**Solución:** Verificar permisos de directorio de sesiones PHP:
```bash
ls -la /var/lib/php/sessions
sudo chmod 1733 /var/lib/php/sessions
```

### Problema 4: "Cannot modify header information"

**Solución:** Asegurar que no hay salida antes de header():
- Verificar que no hay espacios antes de `<?php`
- Verificar encoding UTF-8 sin BOM

### Problema 5: "Call to undefined function password_verify()"

**Solución:** PHP debe ser >= 5.5
```bash
php -v
```

Si es menor, actualizar PHP:
```bash
sudo apt-get update
sudo apt-get install php8.1
```

---

## 🧪 SCRIPT DE PRUEBA COMPLETO

Crear archivo `test_login.php` en la raíz:

```php
<?php
require_once 'config/config.php';

$test_password = 'admin123';
$correct_hash = '$2y$10$eP8F5qGJxGMOZKEXvLlKQ.nLLh9s5OGLcpZPHKxQVJa7OxW6Y8KUq';

echo "=== TEST DE LOGIN ===\n\n";

// Test 1: Verificar hash
echo "1. Verificando hash con contraseña...\n";
if (password_verify($test_password, $correct_hash)) {
    echo "   ✓ Hash válido\n\n";
} else {
    echo "   ✗ Hash inválido\n\n";
}

// Test 2: Consultar usuario en BD
echo "2. Consultando usuario en base de datos...\n";
try {
    $db = Database::getInstance();
    $user = $db->selectOne("SELECT * FROM usuarios WHERE username = 'admin'");
    
    if ($user) {
        echo "   ✓ Usuario encontrado\n";
        echo "   - Username: " . $user['username'] . "\n";
        echo "   - Email: " . $user['email'] . "\n";
        echo "   - Hash length: " . strlen($user['password']) . "\n";
        echo "   - Hash prefix: " . substr($user['password'], 0, 7) . "\n\n";
        
        // Test 3: Verificar contraseña con hash de BD
        echo "3. Verificando contraseña con hash de BD...\n";
        if (password_verify($test_password, $user['password'])) {
            echo "   ✓✓✓ TODO CORRECTO - El login funcionará\n\n";
        } else {
            echo "   ✗ La contraseña no coincide con el hash de la BD\n";
            echo "   SOLUCIÓN: Ejecutar fix_password.php\n\n";
        }
    } else {
        echo "   ✗ Usuario no encontrado\n\n";
    }
} catch (Exception $e) {
    echo "   ✗ Error: " . $e->getMessage() . "\n\n";
}

echo "=== FIN DEL TEST ===\n";
?>
```

Ejecutar:
```bash
php test_login.php
```

---

## 📞 SOPORTE ADICIONAL

Si después de seguir todos estos pasos el problema persiste:

1. Verificar logs de PHP:
   ```bash
   tail -f /var/log/apache2/error.log
   # o
   tail -f logs/php_errors.log
   ```

2. Verificar logs de MySQL:
   ```bash
   tail -f /var/log/mysql/error.log
   ```

3. Habilitar debug en config.php:
   ```php
   error_reporting(E_ALL);
   ini_set('display_errors', 1);
   ```

---

## ✅ CHECKLIST DE VERIFICACIÓN

- [ ] Base de datos `certificados_residencia` existe
- [ ] Tabla `usuarios` tiene registros
- [ ] Usuario `admin` existe en la tabla
- [ ] Hash de contraseña tiene 60 caracteres
- [ ] Hash comienza con `$2y$10$`
- [ ] PHP versión >= 8.0
- [ ] Extensión PDO MySQL habilitada
- [ ] Permisos de archivos correctos (755/644)
- [ ] Sin espacios antes de `<?php` en archivos
- [ ] Sesiones PHP funcionando

---

## 🎯 RESUMEN

**La solución más rápida y efectiva:**

1. Ejecutar: `database/fix_password.php`
2. Login con: `admin` / `admin123`
3. ¡Listo! 🎉

**Hash correcto a usar:**
```
$2y$10$eP8F5qGJxGMOZKEXvLlKQ.nLLh9s5OGLcpZPHKxQVJa7OxW6Y8KUq
```

Este hash corresponde exactamente a la contraseña: `admin123`

---

**Última actualización:** Noviembre 2025  
**Versión:** 1.0.0
