# Documentación Técnica - Sistema de Certificados de Residencia

## Arquitectura del Sistema

### Stack Tecnológico

- **Backend**: PHP 8.0+
- **Base de Datos**: MySQL 5.7+
- **Frontend**: Bootstrap 5, jQuery, AJAX
- **Generación PDF**: TCPDF
- **Firma Digital**: FPDI + OpenSSL
- **Email**: PHPMailer
- **Gestión de Dependencias**: Composer

### Patrón de Diseño

El sistema utiliza un patrón **MVC modificado** con las siguientes capas:

1. **Modelo (src/)**: Clases PHP que representan la lógica de negocio
2. **Vista (public/)**: Páginas HTML/PHP para la interfaz de usuario
3. **Controlador (api/)**: Endpoints REST para operaciones AJAX
4. **Configuración (config/)**: Archivos de configuración central

## Componentes Principales

### 1. Database.php
Singleton para manejo de conexiones PDO con MySQL.

**Métodos principales:**
- `getInstance()`: Obtener instancia única
- `query()`: Ejecutar consultas preparadas
- `select()`: SELECT con parámetros
- `insert()`: INSERT con datos asociativos
- `update()`: UPDATE con condiciones
- `delete()`: DELETE con condiciones

### 2. Auth.php
Manejo de autenticación y autorización.

**Funcionalidades:**
- Login/logout de usuarios
- Validación de sesiones
- Timeout de inactividad
- Generación de tokens CSRF
- Registro de actividad

### 3. Solicitud.php
Gestión del ciclo de vida de solicitudes.

**Métodos:**
- `crear()`: Nueva solicitud con validación de RUT
- `obtenerTodas()`: Lista con filtros
- `obtenerPorId()`: Detalle completo
- `actualizarEstado()`: Cambio de estado
- `obtenerEstadisticas()`: Métricas del sistema

### 4. CertificadoPDF.php
Generación de certificados en PDF.

**Proceso:**
1. Obtener datos de solicitud
2. Cargar configuración municipal
3. Generar PDF con TCPDF
4. Aplicar formato oficial
5. Calcular hash del documento
6. Guardar en sistema de archivos

### 5. Mailer.php
Envío de correos electrónicos.

**Templates:**
- Confirmación de solicitud
- Envío de certificado
- Notificaciones administrativas

## API REST

### Endpoints de Solicitudes

**GET /api/solicitudes.php**
- Listar todas las solicitudes (autenticado)
- Parámetros: estado, busqueda, fecha_desde, fecha_hasta
- Respuesta: JSON con array de solicitudes

**GET /api/solicitudes.php?id={id}**
- Obtener solicitud específica (autenticado)
- Respuesta: JSON con datos completos

**POST /api/solicitudes.php**
- Crear nueva solicitud (público)
- Body: FormData con campos y archivos
- Validaciones: RUT, email, reCAPTCHA
- Respuesta: JSON con número de solicitud

**PUT /api/solicitudes.php**
- Actualizar estado (autenticado)
- Body: id, estado, observaciones
- Respuesta: JSON con resultado

### Endpoints de Certificados

**POST /api/certificados.php?action=generar**
- Generar PDF del certificado
- Body: solicitud_id
- Respuesta: JSON con ruta del PDF

**POST /api/certificados.php?action=firmar**
- Firmar certificado digitalmente
- Body: certificado_id, certificado_pfx (file), password_pfx
- Proceso: Cargar PFX, aplicar firma, actualizar BD
- Respuesta: JSON con resultado

**POST /api/certificados.php?action=enviar**
- Enviar certificado por email
- Body: certificado_id
- Proceso: Adjuntar PDF, enviar SMTP, actualizar estado
- Respuesta: JSON con resultado

**GET /api/certificados.php?action=descargar&certificado_id={id}**
- Descargar PDF del certificado
- Headers: application/pdf, Content-Disposition attachment
- Log: Registra descarga en auditoría

## Base de Datos

### Tabla: usuarios
Administradores del sistema.
- `id`: Primary key
- `username`: Único, para login
- `password`: Hash bcrypt
- `nombre_completo`: Nombre del funcionario
- `email`: Correo electrónico
- `cargo`: Título/puesto
- `activo`: Estado de la cuenta

### Tabla: solicitudes
Solicitudes de certificados.
- `id`: Primary key
- `numero_solicitud`: Único, formato SOL-YYYY-#####
- `nombre_completo`, `rut`, `direccion`: Datos del solicitante
- `telefono`, `email`: Contacto
- `motivo`: Razón de la solicitud
- `estado`: ENUM(pendiente, en_proceso, aprobada, rechazada, enviada)
- `fecha_solicitud`: Timestamp de creación
- `ip_solicitante`: IP del cliente

### Tabla: adjuntos
Archivos subidos con solicitudes.
- `id`: Primary key
- `solicitud_id`: FK a solicitudes
- `nombre_archivo`: Original filename
- `ruta_archivo`: Path en filesystem
- `tipo_mime`: Content-type
- `tamano_bytes`: File size

### Tabla: certificados
Certificados generados.
- `id`: Primary key
- `solicitud_id`: FK a solicitudes
- `numero_certificado`: Único, formato CERT-YYYY-#####
- `ruta_pdf`: Path del PDF
- `hash_documento`: SHA-256 del archivo
- `firmado`: Boolean
- `fecha_generacion`, `fecha_firma`: Timestamps
- `enviado_correo`: Boolean
- `fecha_envio_correo`: Timestamp

### Tabla: logs_actividad
Auditoría del sistema.
- `id`: Primary key
- `usuario_id`: FK a usuarios (nullable)
- `solicitud_id`: FK a solicitudes (nullable)
- `accion`: Tipo de evento
- `descripcion`: Detalle del evento
- `ip_address`, `user_agent`: Datos del cliente
- `fecha_hora`: Timestamp

### Tabla: configuracion
Parámetros del sistema.
- `clave`: Identificador único
- `valor`: Contenido
- `descripcion`: Documentación
- `tipo`: ENUM(texto, numero, booleano, json)

## Seguridad

### Autenticación
- Sesiones seguras con cookie HttpOnly y SameSite
- Contraseñas hasheadas con bcrypt (cost 10)
- Timeout de inactividad (1 hora por defecto)
- Regeneración de ID de sesión en login

### Prevención de Ataques

**SQL Injection**
- Uso exclusivo de prepared statements (PDO)
- Validación de tipos de datos

**XSS (Cross-Site Scripting)**
- Escape de output con htmlspecialchars()
- Content-Security-Policy headers
- Sanitización de inputs

**CSRF (Cross-Site Request Forgery)**
- Tokens CSRF en formularios
- Validación en operaciones POST/PUT/DELETE

**File Upload Attacks**
- Whitelist de extensiones permitidas
- Validación de MIME types
- Límite de tamaño (5MB)
- Nombres de archivo únicos (uniqid + timestamp)

**Session Hijacking**
- Cookies con flag Secure en HTTPS
- IP tracking opcional
- User-Agent validation

### Headers de Seguridad
```php
X-Frame-Options: SAMEORIGIN
X-Content-Type-Options: nosniff
X-XSS-Protection: 1; mode=block
Referrer-Policy: strict-origin-when-cross-origin
```

## Firma Digital

### Proceso de Firma

1. **Carga del Certificado**
   - Usuario sube archivo .pfx/.p12
   - Se guarda temporalmente

2. **Validación**
   - Verificar contraseña del certificado
   - Validar vigencia y emisor

3. **Aplicación de Firma**
   ```php
   $pdf->setSignature($certificateData, $certificateData, $password, '', 2, $info);
   ```

4. **Almacenamiento**
   - Sobrescribir PDF original
   - Actualizar flag `firmado` en BD
   - Registrar timestamp de firma

5. **Verificación**
   - Hash SHA-256 del documento
   - Metadatos de firma en PDF

### Formatos Soportados
- PKCS#12 (.pfx, .p12)
- Certificados de autoridades chilenas (E-Certchile, Gobierno)

## Flujo de Trabajo

### Solicitud de Certificado (Usuario)

1. Usuario accede al formulario público
2. Completa datos personales
3. Valida RUT automáticamente
4. Adjunta documentos (opcional)
5. Completa reCAPTCHA
6. Acepta términos y envía
7. Sistema valida y crea registro
8. Envía email de confirmación
9. Usuario recibe número de solicitud

### Procesamiento (Administrador)

1. Login al panel de administración
2. Visualiza dashboard con estadísticas
3. Accede a lista de solicitudes
4. Filtra por estado/fecha/búsqueda
5. Abre solicitud específica
6. Revisa datos y adjuntos
7. Cambia estado a "En proceso"
8. Genera certificado PDF
9. Opcionalmente firma digitalmente
10. Envía por correo electrónico
11. Sistema actualiza estado a "Enviada"

## Configuración Avanzada

### SMTP para Gmail

```sql
UPDATE configuracion SET valor = 'smtp.gmail.com' WHERE clave = 'smtp_host';
UPDATE configuracion SET valor = '587' WHERE clave = 'smtp_port';
UPDATE configuracion SET valor = 'tls' WHERE clave = 'smtp_secure';
UPDATE configuracion SET valor = 'tu_email@gmail.com' WHERE clave = 'smtp_user';
UPDATE configuracion SET valor = 'contraseña_de_app' WHERE clave = 'smtp_password';
```

**Nota**: Debe habilitar "Acceso de apps menos seguras" o generar "Contraseña de aplicación".

### reCAPTCHA v2

1. Obtener claves en: https://www.google.com/recaptcha/admin
2. Seleccionar reCAPTCHA v2 "No soy un robot"
3. Agregar dominio autorizado
4. Actualizar en BD:

```sql
UPDATE configuracion SET valor = 'SITE_KEY' WHERE clave = 'captcha_site_key';
UPDATE configuracion SET valor = 'SECRET_KEY' WHERE clave = 'captcha_secret_key';
```

### Personalización de Certificados

Editar `src/CertificadoPDF.php` método `generarContenido()`:

- **Logo**: Agregar imagen en membrete
- **Formato**: Ajustar fonts, colores, espaciado
- **Texto legal**: Modificar declaraciones
- **Firma**: Personalizar cuadro de firma

### Backup Automatizado

Crear script cron:

```bash
#!/bin/bash
# backup.sh

TIMESTAMP=$(date +%Y%m%d_%H%M%S)
mysqldump -u root -p certificados_residencia > backup_$TIMESTAMP.sql
tar -czf uploads_$TIMESTAMP.tar.gz uploads/
```

Programar con crontab:
```
0 2 * * * /path/to/backup.sh
```

## Troubleshooting

### Error: "Class 'TCPDF' not found"
**Solución**: Ejecutar `composer install`

### Error: "Permission denied" en uploads/
**Solución**: `chmod -R 777 uploads/`

### Correos no se envían
**Diagnóstico**:
1. Verificar credenciales SMTP
2. Probar con telnet: `telnet smtp.gmail.com 587`
3. Revisar logs: `tail -f logs/php_errors.log`
4. Verificar firewall permite puerto 587

### PDFs en blanco
**Diagnóstico**:
1. Verificar extensión mbstring: `php -m | grep mbstring`
2. Incrementar memory_limit en php.ini
3. Revisar logs de TCPDF en directorio temporal

### Sesión expira inmediatamente
**Solución**:
1. Verificar permisos en directorio de sesiones
2. Ajustar SESSION_LIFETIME en config.php
3. Verificar configuración de cookies en navegador

## Performance

### Optimizaciones Aplicadas

- **Base de Datos**: Índices en columnas de búsqueda frecuente
- **Archivos**: Cache de configuración en memoria
- **Frontend**: Minificación de CSS/JS (producción)
- **Servidor**: GZIP compression para assets estáticos
- **PDFs**: Generación asíncrona para solicitudes masivas

### Escalabilidad

Para entornos de alto tráfico:

1. **Load Balancing**: Múltiples servidores web
2. **Database**: Replicación MySQL master-slave
3. **File Storage**: NFS o S3 para archivos adjuntos
4. **Cache**: Redis/Memcached para sesiones
5. **CDN**: CloudFlare para assets estáticos

## Mantenimiento

### Tareas Regulares

**Diarias**:
- Revisar logs de errores
- Monitorear espacio en disco

**Semanales**:
- Backup de base de datos
- Backup de archivos uploads/

**Mensuales**:
- Actualizar dependencias: `composer update`
- Revisar y limpiar logs antiguos
- Analizar estadísticas de uso

**Trimestrales**:
- Actualizar certificados SSL
- Renovar certificados digitales de firma
- Auditoría de seguridad

## Contacto y Soporte

**Desarrollador**: Sistema Municipal  
**Email**: soporte@ejemplo.com  
**Teléfono**: +56 45 2491100  
**Horario**: Lunes a Viernes, 9:00 - 18:00

---

**Versión**: 1.0.0  
**Última Actualización**: Noviembre 2025
